<?php

namespace Booyah\ApiManagerClient;

use Booyah\ApiManagerClient\Client\Client;
use Booyah\ApiManagerClient\Exceptions\ClientAuthenticationException;

class ApiManger
{
    public static $API_VERSION = 1;

    /**
     * @var string
     */
    protected $domain;

    /**
     * @var string
     */
    protected $appShortage;

    /**
     * @var string
     */
    protected $companyShortage;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $token;

    /**
     *
     * @param string $managerDomain Domain where to reach api manager
     * @param boolean $enableDevMode enable bebug information
     * @return void
     */
    public function __construct($managerDomain, $companyShortage, $appShortage, $enableDevMode = false)
    {
        $this->domain = $managerDomain;
        $this->appShortage = $appShortage;
        $this->companyShortage = $companyShortage;
        $this->isDev = $enableDevMode;
        $this->init();
    }

    public function setToken($token):self
    {
        $this->token = $token;
        $this->client->addHeaders(['token' => $token]);

        return $this;
    }

    /**
     * @param string $endpoint Final endpoint e.g user.check
     * @param array $data
     *
     * @throws ClientAuthenticationException
     *
     * @return array
     */
    public function get($endpoint, array $data) :array
    {
        if (! $this->token) {
            throw new ClientAuthenticationException('Missing auth token');
        }

        $point = 'get/'.$endpoint.'?'.\http_build_query($data);
        $response = $this->client->setMethod('GET')
            ->setEndpoint($point)
            ->sendRaw();

        return $this->handleResponse($response);
    }

    /**
     * @param string $endpoint Final endpoint e.g user.check
     * @param array $data
     *
     * @throws ClientAuthenticationException
     *
     * @return array
     */
    public function send($endpoint, array $data) :array
    {
        if (! $this->token) {
            throw new ClientAuthenticationException('Missing auth token');
        }
        $response = $this->client->setMethod('POST')
            ->addPayload($data)
            ->setEndpoint('send/'.$endpoint)
            ->sendRaw();

        return $this->handleResponse($response);
    }

    protected function handleResponse($response)
    {
        $content = $response->getContent();
        $code = $response->getStatusCode();

        return [
            'code' => $code,
            'error' => $response->isSuccessful(),
            'reason' => $response->getReasonPhrase(),
            'data' => $content,
        ];
    }

    private function init()
    {
        $url = $this->makeBaseUrl();
        $options = [
            'baseUrl' => $url
        ];

        $this->client = new Client($options);
    }

    private function makeBaseUrl()
    {
        $protocol = 'https://';
        if ($this->isDev) {
            $protocol = 'http://';
        }

        return $protocol.$this->domain.'/api/v'.self::$API_VERSION.'/internal/'.$this->companyShortage.'/'.$this->appShortage.'/';
    }
}
