<?php

namespace Booyah\ApiManagerClient\Client;

use Booyah\ApiManagerClient\Exceptions\ClientAuthenticationException;
use Booyah\ApiManagerClient\Exceptions\MissingParameterException;
use GuzzleHttp\Client as GuzzleClient;
use Redtree\Crypt\Decryption;
use Redtree\Crypt\Encryption;

class Client
{
    /**
     * @var mixed string
     */
    protected $baseUrl;

    /**
     * @var mixed string
     */
    protected $method;

    /**
     * @var mixed string
     */
    protected $endpoint;

    /**
     * @var array
     */
    protected $headers;

    /**
     * @var array
     */
    protected $payload;

    public function __construct(array $options = [])
    {
        $this->baseUrl = array_get($options, 'baseUrl');
        $this->method = array_get($options, 'method', 'get');
        $this->endpoint = array_get($options, 'endpoint');
        $this->options = $options;
        $this->headers = [];
        $this->payload = [];
    }

    /**
     * @param string $method
     * @return Client
     */
    public function setMethod($method) : Client
    {
        $this->method = $method;

        return $this;
    }

    /**
     * @param string $endpoint
     * @return Client
     */
    public function setEndpoint($endpoint) : Client
    {
        $this->endpoint = $endpoint;

        return $this;
    }

    /**
     * @param array $headers
     * @return Client
     */
    public function setHeaders(array $headers) : Client
    {
        $this->headers = $headers;

        return $this;
    }

    /**
     * @param array $headers
     * @return Client
     */
    public function addHeaders(array $headers) : Client
    {
        $this->headers = array_merge($this->headers, $headers);

        return $this;
    }

    /**
     * @param array $payload
     * @return Client
     */
    public function setPayload(array $payload) : Client
    {
        $this->payload = $payload;

        return $this;
    }

    /**
     * @param array $payload
     * @return Client
     */
    public function addPayload(array $payload) : Client
    {
        $this->payload = array_merge($this->payload, $payload);

        return $this;
    }

    /**
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \Exception
     */
    public function sendRaw()
    {
        if (empty($this->baseUrl)) {
            throw new MissingParameterException('Missing url base.');
        }

        if (empty($this->endpoint)) {
            throw new MissingParameterException('Missing endpoint for request.');
        }

        $client = new GuzzleClient(array_merge($this->options, ['http_errors' => false, 'base_uri' => $this->baseUrl]));

        $response = $client->request($this->method, $this->endpoint, ['headers' => $this->headers, 'form_params' => $this->payload, 'json' => $this->payload]);

        return new Response(
            $response->getStatusCode(),
            $response->getHeaders(),
            $response->getBody(),
            $response->getProtocolVersion(),
            $response->getReasonPhrase()
        );
    }

    public function sendCrypted()
    {
        if (empty($this->baseUrl)) {
            throw new MissingParameterException('Missing url base.');
        }

        if (empty($this->endpoint)) {
            throw new MissingParameterException('Missing endpoint for request.');
        }

        if (! array_get($this->headers, 'Authorization')) {
            throw new ClientAuthenticationException('Missing Authorization Header');
        }

        $crypted = (new Encryption())
            ->setKey(array_get($this->headers, 'Authorization'))
            ->encrypt($this->payload)
            ->get();

        $this->addHeaders(['X-Proto-Encrypt' => '1']);
        $client = new GuzzleClient(array_merge($this->options, ['http_errors' => false, 'base_uri' => $this->baseUrl]));
        $response = $client->request($this->method, $this->endpoint, ['headers' => $this->headers, 'json' => $crypted]);

        $responseNew = new Response(
            $response->getStatusCode(),
            $response->getHeaders(),
            $response->getBody(),
            $response->getProtocolVersion(),
            $response->getReasonPhrase()
        );

        if ($responseNew->isSuccessful()) {
            $content = $requestNew->getBody()->getContents();
            try {
                $decrypted = (new Decryption())
                ->setKey(array_get($this->headers, 'Authorization'))
                ->decrypt($content)
                ->get();
            } catch (\InvalidArgumentException $e) {
                $decrypted = null;
            }
            $responseNew->bodyOverwrite(json_encode($decrypted));
        }

        return $responseNew;
    }

    /**
     * @param string $baseUrl
     * @return Client
     */
    public function setBaseUrl($baseUrl) : Client
    {
        $this->baseUrl = $baseUrl;

        return $this;
    }
}
