<?php

namespace booyah\WOPP\Models;

use booyah\WOPP\Helpers\Arrayable;
use booyah\WOPP\Helpers\Collection;
use booyah\WOPP\Helpers\Jsonable;
use booyah\WOPP\Helpers\Str;
use booyah\WOPP\Models\Traits\GuardAttributes;
use booyah\WOPP\Models\Traits\HasAttributes;
use Exception;

class Model implements ArrayAble, Jsonable
{
    use HasAttributes,
        GuardAttributes;

    const PREFIX = 'booyah_wopp_';

    /**
     * @var string
     */
    const CREATED_AT = 'created_at';

    /**
     * @var string
     */
    const UPDATED_AT = 'updated_at';

    /**
     * @var bool
     */
    public $exists = false;

    /**
     * @var bool
     */
    public $wasRecentlyCreated = false;

    /**
     * @var string
     */
    protected $table;

    /**
     * @var string
     */
    protected $primaryKey = 'id';

    /**
     * @var string
     */
    protected $keyType = 'int';

    protected $wpDB;

    /**
     * @var array
     */
    protected $wheres = [];

    public function __construct(array $attributes = [])
    {
        global $wpdb;

        $this->wpDB = $wpdb;

        $this->fill($attributes);
    }

    /**
     * @param string $key
     * @return mixed
     */
    public function __get($key)
    {
        return $this->getAttribute($key);
    }

    /**
     * @param string $key
     * @param mixed $value
     */
    public function __set($key, $value)
    {
        $this->setAttribute($key, $value);
    }

    /**
     * @param string $key
     * @return bool
     */
    public function __isset($key)
    {
        return ! is_null($this->getAttribute($key));
    }

    /**
     * @param string $key
     */
    public function __unset($key)
    {
        unset($this->attributes[$key]);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function __toString()
    {
        return $this->toJson();
    }

    /**
     * @param string $method
     * @param array $parameters
     * @return mixed
     */
    public static function __callStatic($method, $parameters)
    {
        return (new static)->$method(...$parameters);
    }

    /**
     * Handle dynamic method calls into the method.
     *
     * @param  string  $method
     * @param  array   $parameters
     * @return mixed
     *
     */
    public function __call($method, $parameters)
    {
        if (Str::startsWith($method, 'where')) {
            return $this->dynamicWhere($method, $parameters);
        }

        switch ($method) {
            case 'get':
                return $this->get();
            case 'where':
                return $this->where(...$parameters);
            case 'find':
                return $this->find(...$parameters);
        }

        return null;
    }

    /**
     * @return string
     */
    protected function getTable()
    {
        if (! isset($this->table)) {
            return $this->wpDB->prefix . self::PREFIX . str_replace(
                '\\', '', Str::snake(class_basename($this) . 's')
            );
        }

        return $this->wpDB->prefix . self::PREFIX . $this->table;
    }

    /**
     * @return string
     */
    public function getKeyName()
    {
        return $this->primaryKey;
    }

    /**
     * @return string
     */
    public function getKeyType()
    {
        return $this->keyType;
    }

    /**
     * @return mixed
     */
    public function getKey()
    {
        return $this->getAttribute($this->getKeyName());
    }

    /**
     * @return array
     */
    public function toArray()
    {
        return $this->attributesToArray();
    }

    /**
     * @param int $options
     * @return false|string
     * @throws \Exception
     */
    public function toJson($options = 0)
    {
        $json = json_encode($this->toArray(), $options);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \Exception(json_last_error_msg());
        }

        return $json;
    }

    /**
     * @param array $attributes
     * @return $this
     */
    public function fill(array $attributes)
    {
        foreach ($attributes as $key => $value) {
            if ($this->isFillable($key)) {
                $this->setAttribute($key, $value);
            }
        }

        return $this;
    }

    /**
     * @return bool
     */
    public function save()
    {
        if ($this->exists) {
            $saved = $this->performUpdate();
        } else {
            $saved = $this->performInsert();
        }

        return $saved;
    }

    public function update(array $attributes)
    {
        return $this->fill($attributes)->save();
    }

    /**
     * @param array $attributes
     */
    public function fillFromDatabase(array $attributes = [])
    {
        foreach ($attributes as $key => $value) {
            $this->setAttribute($key, $value);
        }

        $this->exists = true;
    }

    /**
     * Delete the model from the database.
     *
     * @return bool|null
     *
     * @throws \Exception
     */
    public function delete()
    {
        if (is_null($this->getKeyName())) {
            throw new Exception('No primary key defined on model.');
        }

        // If the model doesn't exist, there is nothing to delete so we'll just return
        // immediately and not do anything else. Otherwise, we will continue with a
        // deletion process on the model, firing the proper events, and so forth.
        if (! $this->exists) {
            return;
        }

        $this->performDeleteOnModel();

        return true;
    }

    /**
     * Force a hard delete on a soft deleted model.
     *
     * This method protects developers from running forceDelete when trait is missing.
     *
     * @return bool|null
     * @throws Exception
     */
    public function forceDelete()
    {
        return $this->delete();
    }

    /**
     * Handles dynamic "where" clauses to the query.
     *
     * @param  string  $method
     * @param  mixed  $parameters
     * @return $this
     */
    public function dynamicWhere($method, $parameters)
    {
        $finder = substr($method, 5);

        $segments = preg_split(
            '/(And|Or)(?=[A-Z])/', $finder, -1, PREG_SPLIT_DELIM_CAPTURE
        );

        // The connector variable will determine which connector will be used for the
        // query condition. We will change it as we come across new boolean values
        // in the dynamic method strings, which could contain a number of these.
        $connector = 'and';

        $index = 0;

        foreach ($segments as $segment) {
            // If the segment is not a boolean connector, we can assume it is a column's name
            // and we will add it to the query as a new constraint as a where clause, then
            // we can keep iterating through the dynamic method string's segments again.
            if ($segment !== 'And' && $segment !== 'Or') {
                $this->addDynamic($segment, $connector, $parameters, $index);

                $index++;
            }

            // Otherwise, we will store the connector so we know how the next where clause we
            // find in the query should be connected to the previous ones, meaning we will
            // have the proper boolean connector to connect the next where clause found.
            else {
                $connector = $segment;
            }
        }

        return $this;
    }

    /**
     * @param $column
     * @param $operator
     * @param $value
     * @return $this
     */
    protected function where($column, $operator, $value = null)
    {
        if ($value === null) {
            $value = $operator;
            $operator = '=';
        }

        $this->wheres[] = [
            'column' => $column,
            'operator' => $operator,
            'value' => $value,
        ];

        return $this;
    }

    /**
     * @param mixed $value
     * @return self
     */
    protected function find($value)
    {
        return $this->where($this->primaryKey, $value)->get()->first();
    }

    protected function all()
    {
        return $this->get();
    }

    protected function get()
    {
        $query = 'SELECT * FROM ' . $this->getTable();

        if (count($this->wheres) > 0) {
            $query .= ' WHERE ';

            for ($i = 0; $i < count($this->wheres); $i++) {
                if ($i > 0) {
                    $query .= ' AND ';
                }

                $query .= $this->wheres[$i]['column'] . ' ' . $this->wheres[$i]['operator'] . ' ' . $this->wheres[$i]['value'];
            }

            $this->wheres = [];
        }

        $models = $this->wpDB->get_results($query, ARRAY_A);

        $collection = new Collection();

        foreach ($models as $model) {
            $tmp = new static();
            $tmp->fillFromDatabase($model);

            $collection->push($tmp);
        }

        return $collection;
    }

    /**
     * @return bool
     */
    protected function performInsert()
    {
        $saved = $this->wpDB->insert($this->getTable(), $this->getAttributes());

        if ($saved) {
            $this->setAttribute($this->getKeyName(), $this->wpDB->insert_id);
            $this->exists = true;
            $this->wasRecentlyCreated = true;
        }

        return $saved;
    }

    /**
     * @return bool
     */
    protected function performUpdate()
    {
        return $this->wpDB->update($this->getTable(), array_merge($this->getAttributes(), [static::UPDATED_AT => date('Y-m-d H:i:s')]), [$this->getKeyName() => $this->getKey()]);
    }

    /**
     * Perform the actual delete query on this model instance.
     *
     * @return void
     */
    protected function performDeleteOnModel()
    {
        $this->exists = false;

        $this->wpDB->delete($this->getTable(), [$this->getKeyName() => $this->getKey()]);
    }

    /**
     * Add a single dynamic where clause statement to the query.
     *
     * @param  string  $segment
     * @param  string  $connector
     * @param  array   $parameters
     * @param  int     $index
     * @return void
     */
    protected function addDynamic($segment, $connector, $parameters, $index)
    {
        // Once we have parsed out the columns and formatted the boolean operators we
        // are ready to add it to this query as a where clause just like any other
        // clause on the query. Then we'll increment the parameter index values.
        $bool = strtolower($connector);

        $this->where(Str::snake($segment), '=', $parameters[$index], $bool);
    }
}
