<?php

namespace booyah\WOPP\Services;

use DateInterval;
use DateTime;

class ChangeDateService
{
    /**
     * Wechseldaten
     *
     * @var array
     */
    protected $changeDates = array();

    /**
     * Startdatum ab wann die Berechnung gilt
     *
     * @var Datetime
     */
    protected $start;

    /**
     * Informationen zu den Perioden
     *
     * @var array
     */
    protected $period = array();

    /**
     * Offset für Startdatum
     *
     * @var array
     */
    protected $offset = array();

    public function __construct()
    {
        $this->start = new DateTime();
    }

    /**
     * Überschreibe das Startdatum
     *
     * @param DateTime $date Default Today
     * @return $this
     */
    public function setStart(DateTime $date)
    {
        $this->start = $date;

        return $this;
    }

    /**
     * Periode zur Berechnung des Datums
     *
     * @param string $userInterval Angabe für das Intervall (day,week,month,year, DateIntervalFormats)
     * @param integer $count Wie oft soll das Intervall durchlaufen werden.
     * @param array/integer $daysOfMonth Zu welchen Tag(en) des Monats soll gewechselt werden dürfen
     * @return $this
     */
    public function setPeriod($userInterval, $count, $daysOfMonth)
    {
        switch ($userInterval) {
            case 'day':
                $interval = 'P1D';
                break;
            case 'week':
                $interval = 'P7D';
                break;
            case 'month':
                $interval = 'P1M';
                break;
            case 'year':
                $interval = 'P1Y';
                break;
            default:
                $interval = $userInterval;
                break;
        }

        if (!is_array($daysOfMonth)) {
            $daysOfMonth = array($daysOfMonth);
        }

        $this->period = array(
            'interval' => $interval,
            'count' => $count,
            'days' => $daysOfMonth,
        );

        return $this;
    }

    /**
     * Falls es einen Offset gibt, diesen hier angeben
     * setOffet('N', 14) ergibt einen Offset von 14 Werktagen.
     *
     * @param string $type Art des Offsets (D: Days, M: Months, Y: Years, N: Weekdays, W: Weeks)
     * @param integer $count Wie groß soll der Offset sein.
     * @return $this
     */
    public function setOffset($type, $count)
    {
        $allowedTypes = array(
            'D',
            'M',
            'Y',
            'N',
        );

        if (! in_array($type, $allowedTypes)) {
            trigger_error("Wrong Type of Parameter", E_USER_ERROR);
        }

        $this->offset = array(
            'type' => $type,
            'count' => $count,
        );

        return $this;
    }

    /**
     * Berechne die möglichen Wechseldaten
     *
     * @return $this
     * @throws \Exception
     */
    public function calculate()
    {

        $this->changeDates = array();

        if (!empty($this->offset)) {
            $this->setStartWithOffset();
        }

        if (empty($this->period)) {
            trigger_error("Missing Period", E_USER_ERROR);
        }

        $this->calculateDate();

        return $this;
    }

    /**
     * Gib alle Daten als Array zurück
     *
     * @return array
     */
    public function getDates()
    {
        return $this->changeDates;
    }

    public function getFormattedDates($format)
    {
        $dates = array();

        foreach ($this->changeDates as $date) {
            array_push($dates, $date->format($format));
        }

        return $dates;
    }

    /**
     * Liefert die Wechsel Daten als String
     *
     * @param string $delimiter
     * @param string $format strftime Format
     * @return string
     */
    public function getDatesAsString($delimiter, $format)
    {
        $dates = array();

        foreach ($this->changeDates as $date) {
            array_push($dates, strftime($format, $date->getTimestamp()));
        }

        return implode($delimiter, $dates);
    }

    /**
     * Addiere den Offset auf das Startdatum
     * @throws \Exception
     */
    private function setStartWithOffset()
    {
        $start = $this->start;

        $counter = 0;
        $weekdays = false;

        switch ($this->offset['type']) {
            case 'D':
                $intervalFormat = 'P1D';
                break;
            case 'N':
                $intervalFormat = 'P1D';
                $weekdays = true;
                break;
            case 'M':
                $intervalFormat = 'P1M';
                break;
            case 'Y':
                $intervalFormat = 'P1Y';
                break;
        }

        $interval = new DateInterval($intervalFormat);

        while ($counter < $this->offset['count']) {
            $start->add($interval);
            $addCount = true;

            if ($weekdays) {
                if ($start->format('N') >= 6) {
                    $addCount = false;
                }
            }

            if ($addCount) {
                $counter +=1;
            }
        }

        $this->start = $start;
    }

    /**
     * Berechne die Daten für den Wechsel
     * @throws \Exception
     */
    private function calculateDate()
    {
        $counter = 0;
        $start = $this->start;
        $start->modify('first day of this month');
        $interval = new DateInterval($this->period['interval']);

        $days = $this->period['days'];

        while ($counter < $this->period['count']) {
            $start->add($interval);
            $year = $start->format('Y');
            $month = $start->format("m");

            foreach ($days as $day) {
                $date = new DateTime();
                $date->setDate($year, $month, $day);
                array_push($this->changeDates, $date);
            }

            $counter++;
        }
    }
}