<?php

namespace booyah\WOPP\Services;

use booyah\WOPP\Models\Model;

class OrderExport
{
    protected $products;

    protected $delimiter;
    protected $enclosure;
    protected $escapeChar;
    protected $useBom = false;

    protected $wpDB;

    protected $headers = [
        'id',
        'product',
        'usage',
        'networknumber',
        'first_name',
        'last_name',
        'salutation',
        'title',
        'birthday',
        'phone_number',
        'email',
        'order_type',
        'meter_number',
        'previous_supplier_code',
        'previous_supplier_name',
        'previous_supplier_customer_number',
        'desired_at',
        'is_cancelled',
        'cancelled_at',
        'moving_in_at',
        'reading_at',
        'reading_at',
        'meter_reading',
        'shipping_postcode',
        'shipping_city',
        'shipping_street',
        'shipping_house_number',
        'is_billing_address_different',
        'billing_first_name',
        'billing_last_name',
        'billing_postcode',
        'billing_city',
        'billing_street',
        'billing_house_number',
        'payment_method',
        'account_owner',
        'iban',
        'bic',
        'bankname',
        'created_at',
    ];

    public function __construct()
    {
        global $wpdb;

        $this->wpDB = $wpdb;

        $this->setOptions();
    }

    /**
     * @param array|int $products
     * @return OrderExport
     */
    public function setProducts($products): self
    {
        $this->products = is_array($products) ? $products : [$products];

        return $this;
    }

    /**
     * @param string $delimiter
     * @param string $enclosure
     * @param string $escapeChar
     * @return OrderExport
     */
    public function setOptions($delimiter = ',', $enclosure = '"', $escapeChar = '\\'): self
    {
        $this->delimiter = $delimiter;
        $this->enclosure = $enclosure;
        $this->escapeChar = $escapeChar;

        return $this;
    }

    /**
     * @return OrderExport
     */
    public function useBom(): self
    {
        $this->useBom = true;

        return $this;
    }

    public function export()
    {
        $queryProducts = 'SELECT id, name FROM ' . $this->wpDB->prefix . Model::PREFIX . 'products';
        $queryProductsWheres = [];

        if ($this->products !== null) {
            $queryProductsWheres[] = 'id IN (' . implode(',', $this->products) . ')';
        }

        if (count($queryProductsWheres) > 0) {
            $queryProductsWhere = ' WHERE ';

            for ($i = 0; $i < count($queryProductsWheres); $i++) {
                if ($i > 0) {
                    $queryProductsWhere .= ' AND ';
                }

                $queryProductsWhere .= $queryProductsWheres[$i];
            }

            $queryProducts .= $queryProductsWhere . ';';
        }

        $productsResults = $this->wpDB->get_results($queryProducts, OBJECT_K);

        $products = [];

        foreach ($productsResults as $result) {
            $products[$result->id] = $result->name;
        }

        $queryOrders = 'SELECT * FROM ' . $this->wpDB->prefix . Model::PREFIX . 'orders WHERE exported_at IS NULL';

        if (count($products) > 0) {
            $queryOrders .= ' AND product_id in (' . implode(',', array_keys($products)) . ');';
        }

        $ordersResults = $this->wpDB->get_results($queryOrders, OBJECT_K);

        $fileName = 'Export-' . date('d-m-Y') . '.csv';

        header( 'Content-Type: text/csv' );
        header( 'Content-Disposition: attachment;filename=' . $fileName);

        $fp = fopen('php://output', 'w');

        if ($this->useBom) {
            fputs($fp, (chr(0xEF) . chr(0xBB) . chr(0xBF)));
        }

        fputcsv($fp, $this->headers, $this->delimiter, $this->enclosure, $this->escapeChar);

        foreach ($ordersResults as $result) {
            $row = [
                $result->id,
                $products[$result->product_id],
                $result->usage,
                $result->networknumber,
                $result->first_name,
                $result->last_name,
                $result->salutation,
                $result->title,
                $result->birthday,
                $result->phone_number,
                $result->email,
                $result->order_type,
                $result->meter_number,
                $result->previous_supplier_code,
                $result->previous_supplier_name,
                $result->previous_supplier_customer_number,
                format_date($result->desired_at),
                $result->is_cancelled,
                format_date($result->cancelled_at),
                format_date($result->moving_in_at),
                format_date($result->reading_at),
                format_date($result->reading_at),
                $result->meter_reading,
                $result->shipping_postcode,
                $result->shipping_city,
                $result->shipping_street,
                $result->shipping_house_number,
                $result->is_billing_address_different,
                $result->billing_first_name,
                $result->billing_last_name,
                $result->billing_postcode,
                $result->billing_city,
                $result->billing_street,
                $result->billing_house_number,
                $result->payment_method,
                $result->account_owner,
                $result->iban,
                $result->bic,
                $result->bankname,
                format_date($result->created_at, 'Y-m-d H:i:s'),
            ];

            fputcsv($fp, $row, $this->delimiter, $this->enclosure, $this->escapeChar);
        }

        fclose($fp);
    }
}