<?php

namespace booyah\WOPP\Services;

use booyah\WOPP\Models\Model;

class OrderExport
{
    protected $products;

    protected $delimiter;
    protected $enclosure;
    protected $escapeChar;
    protected $orderType;
    protected $useBom = false;

    protected $wpDB;

    protected $headers = [
        'ID',
        // Persönliche Daten
        'Kundentyp',
        'Anrede',
        'Titel',
        'Vorname',
        'Nachname',
        'Geburtsdatum',
        'Telefonnummer',
        'E-Mail',
        // Lieferanschrift
        'Lieferadresse Straße',
        'Lieferadresse Hausnummer',
        'Lieferadresse Postleitzahl',
        'Lieferadresse Ort',
        // Rechnungsanschrift
        'Rechnungsadresse',
        'Rechnungsadresse Vorname',
        'Rechnungsadresse Nachname',
        'Rechnungsadresse Straße',
        'Rechnungsadresse Hausnummer',
        'Rechnungsadresse Postleitzahl',
        'Rechnungsadresse Ort',
        // Wechseldaten
        'Zählernummer',
        'Zählerstand',
        'Ablesedatum',
        'Wechselart',
        'Vorversorger',
        'Vorversorger-Code',
        'Vorversorger Kundennummer',
        'Vorversorger bereits gekündigt',
        'Gekündigt zum',
        'Einzugsdatum',
        'Wunschdatum',
        // Zahlungsinformationen
        'Zahlungsmethode',
        'Kontoinhaber',
        'IBAN',
        'BIC',
        'Bankname',
        // Tarifinformationen
        'Produkt ID',
        'Produkt',
        'Netzwerknummer',
        'HTNT',
        'Jahresverbrauch',
        'Arbeitspreis',
        'Jahresverbrauch HT',
        'Arbeitspreis HT',
        'Jahresverbrauch NT',
        'Arbeitspreis NT',
        'Grundpreis',
        'Angezeigter monatlicher Abschlag',
        'Bestellt am',
    ];

    public function __construct()
    {
        global $wpdb;

        $this->wpDB = $wpdb;

        $this->setOptions();
    }

    /**
     * @param array|int $products
     * @return OrderExport
     */
    public function setProducts($products): self
    {
        $this->products = is_array($products) ? $products : [$products];

        return $this;
    }

    /**
     * @param array|int $orderType
     * @return OrderExport
     */
    public function setOrderType($orderType): self
    {
        $this->orderType = $orderType;

        return $this;
    }

    /**
     * @param string $delimiter
     * @param string $enclosure
     * @param string $escapeChar
     * @return OrderExport
     */
    public function setOptions($delimiter = ',', $enclosure = '"', $escapeChar = '\\'): self
    {
        $this->delimiter = $delimiter;
        $this->enclosure = $enclosure;
        $this->escapeChar = $escapeChar;

        return $this;
    }

    /**
     * @return OrderExport
     */
    public function useBom(): self
    {
        $this->useBom = true;

        return $this;
    }

    public function export()
    {
        $queryProducts = 'SELECT id, name FROM ' . $this->wpDB->prefix . Model::PREFIX . 'products';
        $queryProductsWheres = [];

        if ($this->products !== null) {
            $queryProductsWheres[] = 'id IN (' . implode(',', $this->products) . ')';
        }

        if (count($queryProductsWheres) > 0) {
            $queryProductsWhere = ' WHERE ';

            for ($i = 0; $i < count($queryProductsWheres); $i++) {
                if ($i > 0) {
                    $queryProductsWhere .= ' AND ';
                }

                $queryProductsWhere .= $queryProductsWheres[$i];
            }

            $queryProducts .= $queryProductsWhere . ';';
        }

        $productsResults = $this->wpDB->get_results($queryProducts, OBJECT_K);

        $products = [];

        foreach ($productsResults as $result) {
            $products[$result->id] = $result->name;
        }

        $queryOrders = 'SELECT * FROM ' . $this->wpDB->prefix . Model::PREFIX . 'orders';
        if($this->orderType == 1) {
            $queryOrders .= ' WHERE exported_at IS NULL';
        } else if($this->orderType == 2) {
            $queryOrders .= ' WHERE exported_at IS NOT NULL';
        }

        if (count($products) > 0) {
            $queryOrders .= ' ' . (($this->orderType == 1 || $this->orderType == 2) ? 'AND' : 'WHERE') . ' product_id in (' . implode(',', array_keys($products)) . ');';
        }

        $ordersResults = $this->wpDB->get_results($queryOrders, OBJECT_K);

        $fileName = 'Export-' . date('d-m-Y') . '.csv';

        header( 'Content-Type: text/csv' );
        header( 'Content-Disposition: attachment;filename=' . $fileName);

        $fp = fopen('php://output', 'w');

        if ($this->useBom) {
            fputs($fp, (chr(0xEF) . chr(0xBB) . chr(0xBF)));
        }

        fputcsv($fp, $this->headers, $this->delimiter, $this->enclosure, $this->escapeChar);

        foreach ($ordersResults as $result) {
            switch ($result->salutation) {
                case 'male':
                    $customerSalutationReadable = 'Herr';
                    break;
                case 'female':
                    $customerSalutationReadable = 'Frau';
                    break;
                case 'other':
                    $customerSalutationReadable = 'Sonstiges';
                    break;
                default:
                    $customerSalutationReadable = 'Firma';
            }

            switch ($result->order_type) {
                case '0':
                    $orderTypeReadable = 'Versorgerwechsel';
                    break;
                case '1':
                    $orderTypeReadable = 'Umzug';
                    break;
                default:
                    $orderTypeReadable = 'Neueinzug';
            }

            switch ($result->payment_method) {
                case '0':
                    $paymentTypeReadable = 'SEPA-Basislastschrift';
                    break;
                case '1':
                    $paymentTypeReadable = 'Überweisung';
                    break;
                case '2':
                    $paymentTypeReadable = 'Paypal';
                    break;
                default:
                    $paymentTypeReadable = '';
            }

            if($result->usage_price_et > 0) {
                $customerPriceMonthly = (($result->usage_price_et * (int)$result->usage) + ($result->base_price * 12)) / 12;
            } else {
                $customerPriceMonthly = (($result->usage_price_ht * $result->usage) + ($result->usage_price_nt * $result->usage_nt) + ($result->base_price * 12)) / 12;
            }

            $row = [
                $result->id,
                // Persönliche Daten
                $result->customer_type == 0 ? 'Privat' : 'Gewerbe',
                $customerSalutationReadable,
                $result->title,
                $result->first_name,
                $result->last_name,
                format_date($result->birthday),
                $result->phone_number,
                $result->email,
                // Lieferanschrift
                $result->shipping_street,
                $result->shipping_house_number,
                $this->formatPostcode($result->shipping_postcode),
                $result->shipping_city,
                // Rechnungsanschrift
                $result->billing_address_different == 0 ? 'Identisch' : 'Abweichend',
                $result->billing_first_name,
                $result->billing_last_name,
                $result->billing_street,
                $result->billing_house_number,
                $this->formatPostcode($result->billing_postcode),
                $result->billing_city,
                // Wechseldaten
                $result->meter_number,
                $result->meter_reading,
                format_date($result->reading_at),
                $orderTypeReadable,
                $result->previous_supplier_name,
                $result->previous_supplier_code,
                $result->previous_supplier_customer_number,
                $result->is_cancelled == 0 ? 'Nein' : 'Ja',
                format_date($result->cancelled_at),
                format_date($result->moving_in_at),
                format_date($result->desired_at),
                // Zahlungsinformationen
                $paymentTypeReadable,
                $result->account_owner,
                $result->iban,
                $result->bic,
                $result->bankname,
                // Tarifinformationen
                $result->product_id,
                $products[$result->product_id],
                $result->networknumber,
                $result->usage_price_et > 0 ? 'Nein' : 'Ja', // htnt ?
                $result->usage_price_et > 0 ?  $result->usage : null, // Eintarif Verbrauch
                $this->formatNumber($result->usage_price_et), // Eintarif Arbeitspreis
                $result->usage_price_et > 0 ? null : $result->usage, // Doppeltarif HT Verbrauch
                $this->formatNumber($result->usage_price_ht), // Doppeltarif HT Arbeitspreis
                $this->formatNumber($result->usage_nt), // Doppeltarif NT Verbrauch
                $this->formatNumber($result->usage_price_nt), // Doppeltarif NT Arbeitspreis
                $this->formatNumber($result->base_price), // Grundpreis
                $this->formatNumber($customerPriceMonthly),
                format_date($result->created_at, 'd.m.Y H:i:s'),
            ];

            fputcsv($fp, $row, $this->delimiter, $this->enclosure, $this->escapeChar);

            $this->wpDB->update(
                $this->wpDB->prefix . Model::PREFIX . 'orders',
                [ 'exported_at' => date('Y-m-d H:i:s') ],
                [ 'id' => $result->id ]
            );
        }

        fclose($fp);
    }

    /**
     * @param string $postcode
     * @return string
     */
    private function formatPostcode($postcode)
    {
        return str_pad($postcode, 5, '0', STR_PAD_LEFT);
    }

    /**
     * @param string|int|float $number
     * @return mixed
     */
    private function formatNumber($number)
    {
        return str_replace('.', ',', $number);
    }
}