<?php

namespace booyah\WOPP\Services;

use booyah\WOPP\Helpers\Fluent;
use booyah\WOPP\Models\Price;
use booyah\WOPP\Models\Product;

class PriceCalculationService
{
    /**
     * @var Product
     */
    protected $product;

    /**
     * @var int
     */
    protected $usage;

    /**
     * @var string
     */
    protected $date;

    /**
     * @var string
     */
    protected $postcode;

    /**
     * @var int
     */
    protected $customerType = 0;

    /**
     * @var bool
     */
    protected $isHtNt = false;

    /**
     * @var int
     */
    protected $usageNt;

    public function __construct()
    {
        $this->date = date('Y-m-d H:i:s');
    }

    /**
     * @param Product $product
     *
     * @return $this
     */
    public function setProduct(Product $product): self
    {
        $this->product = $product;

        return $this;
    }

    /**
     * @param int $usage
     *
     * @return $this
     */
    public function setUsage($usage): self
    {
        $this->usage = $usage;

        return $this;
    }

    /**
     * @param string $date
     *
     * @return $this
     */
    public function setDate($date): self
    {
        $this->date = $date;

        return $this;
    }

    /**
     *
     * @return $this
     */
    public function setCustomerType($customerType = 0): self
    {
        $this->customerType = $customerType;

        return $this;
    }

    /**
     * @param $postcode
     *
     * @return $this
     */
    public function setPostcode($postcode): self
    {
        $this->postcode = $postcode;

        return $this;
    }

    /**
     * @param $usage
     *
     * @return $this
     */
    public function setUsageNt($usage): self
    {
        $this->usageNt = $usage;
        $this->isHtNt = true;

        return $this;
    }

    public function calculate()
    {
        global $wpdb;

        $ret = new Fluent([
            'usage' => $this->usage,
        ]);

        if ($this->product->customer_type != $this->customerType) {
            return null;
        }

        if ($this->isHtNt) {
            $resultNt = $wpdb->get_row($this->getQuery($this->usageNt, 'nt'), ARRAY_A);
            $resultHt = $wpdb->get_row($this->getQuery($this->usage, 'ht'), ARRAY_A);

            if ($resultNt === null || $resultHt === null) {
                return null;
            }

            $priceHt = new Price();
            $priceHt->fillFromDatabase($resultHt);

            $priceNt = new Price();
            $priceNt->fillFromDatabase($resultNt);


            $ret->basePrice = $priceNt->base_price;
            $ret->usageNt = $this->usageNt;
            $ret->usagePriceHt = $priceHt->usage_price_ht;
            $ret->usagePriceNt = $priceNt->usage_price_nt;
            $ret->yearly = ($ret->usagePriceHt * $this->usage) + ($ret->usagePriceNt * $this->usageNt) + ($ret->basePrice * 12);
            $ret->monthly = ceil($ret->yearly / 12);
        } else {
            $result = $wpdb->get_row($this->getQuery($this->usage, 'et'), ARRAY_A);

            if ($result === null) {
                return null;
            }

            $price = new Price();
            $price->fillFromDatabase($result);

            $ret->basePrice = $price->base_price;
            $ret->usagePrice = $price->usage_price_et;
            $ret->yearly = ($ret->usagePrice * (int) $this->usage) + ($ret->basePrice * 12);
            $ret->monthly = ceil($ret->yearly / 12);
        }

        $ret->networknumber = $price->networknumber;

        return $ret;
    }

    public function getQuery($usage, $type = 'et')
    {
        $query = '
            SELECT * FROM ' . Price::getTable() . '
            WHERE product_id = ' . $this->product->id . '
            AND valid_from <= "' . $this->date . '"
            AND valid_to >= "' . $this->date . '"
            AND application_from <= "' . $this->date . '"
            AND application_to >= "' . $this->date . '"
            AND usage_from <= ' . $usage . '
            AND usage_to >= ' . $usage . '
            AND deleted_at IS NULL 
        ';
        if ($type === 'ht') {
            $query .= '
                AND usage_price_ht > 0.00
            ';
        } elseif ($type === 'nt') {
            $query .= '
                AND usage_price_nt > 0.00
            ';
        } else {
            $query .= '
                AND usage_price_et > 0.00
            ';
        }
        $query .= '
             AND (
                (is_blacklist = 1 AND (postcodes NOT LIKE "%' . $this->postcode . '%" OR postcodes IS NULL))
                OR (is_blacklist = 0 AND (postcodes LIKE "%' . $this->postcode . '%" AND postcodes IS NOT NULL))
            )
            ORDER BY id desc;
        ';

        return $query;
    }
}
