<?php

namespace Redtree\EnetClient;

use Carbon\Carbon;
use GuzzleHttp\Client as GuzzleClient;

class Client
{
    /**
     * @var mixed string
     */
    protected $baseUrl;

    /**
     * @var mixed string
     */
    protected $method;

    /**
     * @var mixed string
     */
    protected $endpoint;

    /**
     * @var array
     */
    protected $headers;

    /**
     * @var array
     */
    protected $payload;

    public function __construct(array $options = [])
    {
        $this->baseUrl = array_get($options, 'baseUrl');
        $this->method = array_get($options, 'method', 'get');
        $this->endpoint = array_get($options, 'endpoint');

        $this->headers = [];
        $this->payload = [];
    }

    /**
     * @param string $baseUrl
     * @return Client
     */
    protected function setBaseUrl($baseUrl) : Client
    {
        $this->baseUrl = $baseUrl;

        return $this;
    }

    /**
     * @param string $method
     * @return Client
     */
    protected function setMethod($method) : Client
    {
        $this->method = $method;

        return $this;
    }

    /**
     * @param string $endpoint
     * @return Client
     */
    protected function setEndpoint($endpoint) : Client
    {
        $this->endpoint = $endpoint;

        return $this;
    }

    /**
     * @param array $headers
     * @return Client
     */
    protected function setHeaders(array $headers) : Client
    {
        $this->headers = $headers;

        return $this;
    }

    /**
     * @param array $headers
     * @return Client
     */
    protected function addHeaders(array $headers) : Client
    {
        $this->headers = array_merge($this->headers, $headers);

        return $this;
    }

    /**
     * @param array $payload
     * @return Client
     */
    protected function setPayload(array $payload) : Client
    {
        $this->payload = $payload;

        return $this;
    }

    /**
     * @param array $payload
     * @return Client
     */
    protected function addPayload(array $payload) : Client
    {
        $this->payload = array_merge($this->payload, $payload);

        return $this;
    }

    /**
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \Exception
     */
    protected function send()
    {
        $cacheString = $this->method . '.' . $this->endpoint;

        foreach ($this->payload as $key => $value) {
            $cacheString .= '.' . $key . '=' . $value;
        }

        $response = cache()->remember($cacheString, Carbon::now()->endOfDay(), function () {
            $client = new GuzzleClient(['http_errors' => false, 'base_uri' => $this->baseUrl]);
            $response = $client->request($this->method, $this->endpoint, ['headers' => $this->headers, 'form_params' => $this->payload, 'json' => $this->payload]);

            return json_decode($response->getBody()->getContents());
        });

        return $response;
    }
}