<?php

namespace Redtree\EnetClient;

use Redtree\EnetClient\Resources\Network\Electricity\NetworkMeasuringPrices;
use Redtree\EnetClient\Resources\Network\Electricity\NetworkNumber;
use Redtree\EnetClient\Resources\Network\Electricity\NetworkUsagePrices;

class EnetClient extends Client
{
    const ENERGY_TYPE_POWER = 'power';
    const ENERGY_TYPE_GAS = 'gas';

    /**
     * @var string
     */
    protected $energyType;

    /**
     * @var array
     */
    protected $data;

    public function __construct(array $options = [])
    {
        parent::__construct($options);

        $this->energyType = self::ENERGY_TYPE_POWER;
        $this->data = [];

        $this->setBaseUrl(config('enetclient.baseUrl'));
        $this->setHeaders(['token' => config('enetclient.token')]);
    }

    /**
     * @param string $type
     * @return EnetClient
     */
    public function setEnergyType($type) : EnetClient
    {
        $this->energyType = $type;

        return $this;
    }

    /**
     * @param array $data
     * @return EnetClient
     */
    public function setData(array $data) : EnetClient
    {
        $this->data = $data;

        return $this;
    }

    /**
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \Exception
     */
    public function getTaxes()
    {
        return $this->addToPayload()->setEndpoint('/api/bocc/taxes')->send();
    }

    /**
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \Exception
     */
    public function getNetworknumber()
    {
        $endpoint = '/api/network/';
        switch ($this->energyType) {
            case self::ENERGY_TYPE_GAS:
                $endpoint .= 'gas';
                break;

            case self::ENERGY_TYPE_POWER:
            default:
                $endpoint .= 'electricity';
                NetworkNumber::compare($this->data);
                break;
        }

        $endpoint .= '/networknumber';

        return $this->addToPayload()->setEndpoint($endpoint)->send();
    }

    /**
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \Exception
     */
    public function getNetworkUsagePrices()
    {
        $endpoint = '/api/network/';
        switch ($this->energyType) {
            case self::ENERGY_TYPE_GAS:
                $endpoint .= 'gas';
                break;

            case self::ENERGY_TYPE_POWER:
            default:
                $endpoint .= 'electricity';
                NetworkUsagePrices::compare($this->data);
                break;
        }

        $endpoint .= '/networkusageprices';

        return $this->addToPayload()->setEndpoint($endpoint)->send();
    }

    /**
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \Exception
     */
    public function getNetworkMeasuringPrices()
    {
        $endpoint = '/api/network/';
        switch ($this->energyType) {
            case self::ENERGY_TYPE_GAS:
                $endpoint .= 'gas';
                break;

            case self::ENERGY_TYPE_POWER:
            default:
                $endpoint .= 'electricity';
                NetworkMeasuringPrices::compare($this->data);
                break;
        }

        $endpoint .= '/measuringprice';

        return $this->addToPayload()->setEndpoint($endpoint)->send();
    }

    protected function addToPayload()
    {
        return $this->addPayload(['energyType' => $this->energyType])->addPayload($this->data);
    }
}
