<?php

namespace Redtree\FeatureFlags;

use App\Models\System\Tenant;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Support\Facades\Log;
use Redtree\FeatureFlags\Models\System\Feature;
use stdClass;
use Throwable;

class FeatureService
{
    protected ?Tenant $tenant = null;

    protected ?Authenticatable $user = null;

    public function __construct()
    {
        $this->tenant = current_tenant();

        if (function_exists('current_user') && current_user() instanceof Authenticatable) {
            $this->user = current_user();
        }
    }

    public function forTenant(Tenant $tenant): self
    {
        $this->tenant = $tenant;

        return $this;
    }

    public function forUser(Authenticatable $user): self
    {
        $this->user = $user;

        return $this;
    }

    /** @throws Throwable */
    public function isActive(string $key): bool
    {
        throw_if($this->tenant === null, new Exception('Missing tenant. Use forTenant() function.'));

        if (class_exists($key)) {
            $key = (new $key())->getKey();
        }

        return cache()
            ->store('array')
            ->rememberForever(sprintf('feature-flags.%s.%s', $key, $this->tenant->uuid), function () use (&$key) {
                return Feature::query()
                    ->where('key', $key)
                    ->whereJsonContains('apps', $this->tenant->getCurrentAppId())
                    ->whereRelation('tenants', function (Builder $builder) {
                        $builder
                            ->where('tenant_id', $this->tenant->id)
                            ->where('feature_tenant.is_active', true);
                    })
                    ->isActive()
                    ->exists();
            });
    }

    public function can(string $featureClass): bool
    {
        if ($this->user === null) {
            return false;
        }

        $featureObject = new $featureClass();

        if (! $featureObject instanceof FeatureInterface) {
            return false;
        }

        if (! $this->isActive($featureObject->getKey())) {
            return false;
        }

        $feature = Feature::where('key', $featureObject->getKey())->first();

        if ($feature === null) {
            return false;
        }

        $roles = $feature
            ->roles()
            ->get();

        if ($roles->isNotEmpty() && ! $this->user->hasRole($roles)) {
            return false;
        }

        $permissions = $feature
            ->permissions()
            ->get();

        if ($permissions->isNotEmpty()) {
            return false;
        }

        return true;
    }

    /** @throws Throwable */
    public function execute(string $featureClass, ...$params): stdClass
    {
        $featureObject = new $featureClass();

        if (! $featureObject instanceof FeatureInterface) {
            return (object) [
                'success' => false,
                'error' => sprintf('Class ´%s´ is not a Featureclass.', $featureClass),
                'data' => null,
            ];
        }

        if (! $this->isActive($featureObject->getKey())) {
            return (object) [
                'success' => false,
                'error' => sprintf('Feature ´%s´ is not active.', $featureObject->getKey()),
                'data' => null,
            ];
        }

        $feature = Feature::where('key', $featureObject->getKey())->first();

        if ($feature === null) {
            return (object) [
                'success' => false,
                'error' => sprintf('Feature ´%s´ does not exists in database.', $featureObject->getKey()),
                'data' => null,
            ];
        }

        if ($this->user !== null) {
            $roles = $feature
                ->roles()
                ->get();

            if ($roles->isNotEmpty() && ! $this->user->hasRole($roles)) {
                return (object) [
                    'success' => false,
                    'error' => sprintf('Current user needs roles ´%s´.', $roles->implode('name', ', ')),
                    'data' => null,
                ];
            }

            $permissions = $feature
                ->permissions()
                ->get();

            if ($permissions->isNotEmpty()) {
                foreach ($permissions as $permission) {
                    if (! $this->user->hasPermissionTo($permission)) {
                        return (object) [
                            'success' => false,
                            'error' => sprintf('Current user needs permission ´%s´.', $permission->name),
                            'data' => null,
                        ];
                    }
                }
            }
        }

        $success = false;
        $data = null;

        try {
            $tmp = $featureObject->execute($params);

            $success = $tmp->success;
            $error = $tmp->error;
            $data = $tmp->data;
        } catch (Exception $exception) {
            Log::error($exception->getMessage());
            Log::error($exception->getTraceAsString());

            $error = $exception;
        }

        return (object) [
            'success' => $success,
            'error' => $error,
            'data' => $data,
        ];
    }
}
