## Installation
 Add the following to the projects composer.json
```json
"repositories": [
    {
        "type":"composer",
        "url":"https://packages.redtree.de"
    }
 ]
```

You can then install the package via composer:
```bash
composer require redtree/feature-flags
```

## Creation

### Create a new Feature Class
Create a new class in app/Features inside the desired app.
```php
<?php

namespace App\Features;

use Redtree\FeatureFlags\FeatureInterface;
use Redtree\FeatureFlags\Traits\BaseFeature;
use stdClass;

class TestFeature implements FeatureInterface
{
    use BaseFeature;

    protected string $key = 'test';

    public function execute(...$params): stdClass
    {
        return (object) [
            'success' => true,
            'error' => null,
            'data' => null,
        ];
    }
}

```
The ``$key`` must be unique across all features.

### Add a new feature for deployment
Inside io.base, extend the DeployFeatures.php in app/Console/Commands/System/Deploy and add the following:
```php
Feature::firstOrCreate([
    'key' => 'test2',
], [
    'apps' => [1, 2],
    'is_core' => false,
    'is_active' => true,
    'is_payment_required' => false,
    'name' => 'Test',
    'description' => 'Test Feature',
    'status' => 'dev',
]);
```
Change values accordingly. 

If you want to add roles and/or permissions store the created feature model and extend the foreach like this:
```php
$testFeature->roles()->firstOrCreate(['app_name' => 'center', 'name' => 'test2']);
$testFeature->permissions()->firstOrCreate(['app_name' => 'center', 'name' => 'feature test2']);
```
The handle method should look like this:

```php
public function handle(): int
{
    $this->info($this->description);

    $testFeature = Feature::firstOrCreate([
        'key' => 'test2',
    ], [
        'apps' => [1, 2],
        'is_core' => false,
        'is_active' => true,
        'is_payment_required' => false,
        'name' => 'Test',
        'description' => 'Test Feature',
        'status' => 'dev',
    ]);

    $tenants = Tenant::all();

    foreach ($tenants as $tenant) {
        $tenant->switchTo();

        app(PermissionsCache::class)->forgetCachedPermissions();

        $testFeature->roles()->firstOrCreate(['app_name' => 'center', 'name' => 'test2']);
        $testFeature->permissions()->firstOrCreate(['app_name' => 'center', 'name' => 'feature test2']);
    }

    return self::SUCCESS;
}
```

## Usage
Inside an app you can run a feature for the current tenant like this:

```php
$data = Feature::execute(TestFeature::class, $params);
```

To check if a feature is active:
```php
Feature::isActive('key');
Feature::isActive(TestFeature::class);
```
To check if multiple Features are active:
```php
Feature::isActive('key', 'key2');
Feature::isActive(['key', 'key2']);
Feature::isActive(TestFeature::class, TestFeature2::class);
Feature::isActive([TestFeature::class, TestFeature2::class]);
```

To check if a feature can run for the current user:
```php
Feature::can(TestFeature::class);
```
