<?php

namespace Redtree\FileLibrary\Media;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Traits\Macroable;
use League\Flysystem\FileExistsException;
use League\Flysystem\FileNotFoundException;
use Redtree\FileLibrary\Exceptions\DiskDoesNotExist;
use Redtree\FileLibrary\Exceptions\FileDoesNotExist;
use Redtree\FileLibrary\Exceptions\FileIsTooBig;
use Redtree\FileLibrary\Exceptions\FileUnacceptableForCollection;
use Redtree\FileLibrary\Exceptions\UnknownType;
use Redtree\FileLibrary\HasMedia;
use Redtree\FileLibrary\Media\File as PendingFile;
use Redtree\FileLibrary\Media\Models\Media;
use Redtree\FileLibrary\Support\File;
use Redtree\FileLibrary\Support\FileNamer;
use Redtree\FileLibrary\Support\RemoteFile;
use Symfony\Component\HttpFoundation\File\File as SymfonyFile;

class FileAdder
{
    use Macroable;

    protected ?Filesystem $filesystem;

    protected ?Model $subject;

    /** @var string|UploadedFile|RemoteFile */
    protected $file;

    protected bool $preserveOriginal = false;

    protected string $pathToFile = '';

    protected string $fileName = '';

    protected string $name = '';

    protected array $manipulations = [];

    protected array $customProperties = [];

    protected array $properties = [];

    protected array $customHeaders = [];

    protected string $visibility = Media::VISIBILITY_PUBLIC;

    public function __construct(Filesystem $fileSystem)
    {
        $this->filesystem = $fileSystem;
    }

    public function setSubject(Model $subject): self
    {
        $this->subject = $subject;

        return $this;
    }

    /**
     * @param string|UploadedFile|RemoteFile $file
     * @throws UnknownType
     */
    public function setFile($file): self
    {
        $this->file = $file;

        if (is_string($file)) {
            $this->pathToFile = $file;
            $this->fileName = pathinfo($file, PATHINFO_BASENAME);
            $this->name = pathinfo($file, PATHINFO_FILENAME);

            return $this;
        }

        if ($file instanceof RemoteFile) {
            $this->pathToFile = $file->getKey();
            $this->fileName = $file->getFilename();
            $this->name = $file->getName();

            return $this;
        }

        if ($file instanceof UploadedFile) {
            $this->pathToFile = $file->getPath() . '/' . $file->getFilename();
            $this->fileName = $file->getClientOriginalName();
            $this->name = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);

            return $this;
        }

        if ($file instanceof SymfonyFile) {
            $this->pathToFile = $file->getPath() . '/' . $file->getFilename();
            $this->fileName = pathinfo($file->getFilename(), PATHINFO_BASENAME);
            $this->name = pathinfo($file->getFilename(), PATHINFO_FILENAME);

            return $this;
        }

        throw UnknownType::create();
    }

    public function preservingOriginal(bool $preserveOriginal = true): self
    {
        $this->preserveOriginal = $preserveOriginal;

        return $this;
    }

    public function usingName(string $name): self
    {
        return $this->setName($name);
    }

    public function setName(string $name): self
    {
        $this->name = $name;

        return $this;
    }

    public function usingFileName(string $fileName): self
    {
        return $this->setFileName($fileName);
    }

    public function setFileName(string $fileName): self
    {
        $this->fileName = $fileName;

        return $this;
    }

    public function withManipulations(array $manipulations): self
    {
        $this->manipulations = $manipulations;

        return $this;
    }

    public function withCustomProperties(array $customProperties): self
    {
        $this->customProperties = $customProperties;

        return $this;
    }

    public function withProperties(array $properties): self
    {
        $this->properties = $properties;

        return $this;
    }

    public function withAttributes(array $properties): self
    {
        return $this->withProperties($properties);
    }

    public function addCustomHeaders(array $customRemoteHeaders): self
    {
        $this->customHeaders = $customRemoteHeaders;

        $this->filesystem->addCustomRemoteHeaders($customRemoteHeaders);

        return $this;
    }

    public function setVisibility(string $visibility): self
    {
        if (in_array($visibility, [Media::VISIBILITY_PUBLIC, Media::VISIBILITY_PRIVATE, Media::VISIBILITY_CUSTOM])) {
            $this->visibility = $visibility;
        }

        return $this;
    }

    /**
     * @throws DiskDoesNotExist
     * @throws FileDoesNotExist
     * @throws FileExistsException
     * @throws FileIsTooBig
     * @throws FileNotFoundException
     * @throws FileUnacceptableForCollection
     */
    public function toMediaCollection(string $collectionName = 'default', string $diskName = ''): Media
    {
        if ($this->file instanceof RemoteFile) {
            $storage = Storage::disk($this->file->getDisk());

            if (! $storage->exists($this->pathToFile)) {
                throw FileDoesNotExist::create($this->pathToFile);
            }

            if ($storage->size($this->pathToFile) > config('file-library.max_file_size')) {
                throw FileIsTooBig::create($this->pathToFile, $storage->size($this->pathToFile));
            }

            $mimeType = $storage->mimeType($this->pathToFile);
            $size = $storage->size($this->pathToFile);
        } else {
            if (! is_file($this->pathToFile)) {
                throw FileDoesNotExist::create($this->pathToFile);
            }

            if (filesize($this->pathToFile) > config('file-library.max_file_size')) {
                throw FileIsTooBig::create($this->pathToFile);
            }

            $mimeType = File::getMimeType($this->pathToFile);
            $size = filesize($this->pathToFile);
        }

        $sanitizedFileName = $this->defaultSanitizer($this->fileName);
        $fileName = app(FileNamer::class)->originalFileName($sanitizedFileName);
        $this->fileName = $this->appendExtension($fileName, pathinfo($sanitizedFileName, PATHINFO_EXTENSION));

        $mediaClass = config('file-library.media_model');
        /** @var Media $media */
        $media = new $mediaClass();
        $media->setConnection($this->subject->getConnectionName());

        $media->disk = $this->determineDiskName($diskName, $collectionName);
        $this->ensureDiskExists($media->disk);

        $media->collection_name = $collectionName;
        $media->name = $this->name;
        $media->file_name = $this->fileName;
        $media->mime_type = $mimeType;
        $media->size = $size;
        $media->manipulations = $this->manipulations;
        $media->custom_properties = $this->customProperties;
        $media->generated_conversions = [];
        $media->visibility = $this->visibility;

        if (filled($this->customHeaders)) {
            $media->setCustomHeaders($this->customHeaders);
        }

        $media->fill($this->properties);

        $this->attachMedia($media);

        return $media;
    }

    protected function defaultSanitizer(string $fileName): string
    {
        return str_replace(['#', '/', '\\', ' '], '-', $fileName);
    }

    protected function appendExtension(string $file, ?string $extension = null): string
    {
        return $extension !== null ? $file . '.' . $extension : $file;
    }

    protected function determineDiskName(string $diskName, string $collectionName): string
    {
        if ($diskName !== '') {
            return $diskName;
        }

        if ($collection = $this->getMediaCollection($collectionName)) {
            $collectionDiskName = $collection->diskName;

            if ($collectionDiskName !== '') {
                return $collectionDiskName;
            }
        }

        return config('file-library.disk_name');
    }

    /**
     * @throws DiskDoesNotExist
     */
    protected function ensureDiskExists(string $diskName): void
    {
        if (config("filesystems.disks.$diskName") === null) {
            throw DiskDoesNotExist::create($diskName);
        }
    }

    /**
     * @throws FileExistsException|FileNotFoundException|FileUnacceptableForCollection
     */
    protected function attachMedia(Media $media): void
    {
        if (! $this->subject->exists) {
            $this->subject->prepareToAttachMedia($media, $this);

            $class = get_class($this->subject);

            $class::created(function ($model) {
                $model->processUnattachedMedia(function (Media $media, self $fileAdder) use (&$model) {
                    $this->processMediaItem($model, $media, $fileAdder);
                });
            });

            return;
        }

        $this->processMediaItem($this->subject, $media, $this);
    }

    /**
     * @throws FileUnacceptableForCollection|FileExistsException|FileNotFoundException
     */
    protected function processMediaItem(HasMedia $model, Media $media, self $fileAdder)
    {
        $this->guardAgainstDisallowedFileAdditions($model, $media);

        $model->media()->save($media);

        if ($fileAdder->file instanceof RemoteFile) {
            $this->filesystem->addRemote($fileAdder->file, $media, $fileAdder->fileName);
        } else {
            $this->filesystem->add($fileAdder->pathToFile, $media, $fileAdder->fileName);
        }

        if (! $fileAdder->preserveOriginal) {
            if ($fileAdder->file instanceof RemoteFile) {
                Storage::disk($fileAdder->file->getDisk())->delete($fileAdder->file->getKey());
            } else {
                unlink($fileAdder->pathToFile);
            }
        }

        if ($collectionSizeLimit = optional($this->getMediaCollection($media->collection_name))->collectionSizeLimit) {
            $collectionMedia = $this->subject->fresh()->getMedia($media->collection_name);

            if ($collectionMedia->count() > $collectionSizeLimit) {
                $model->clearMediaCollectionExcept($media->collection_name, $collectionMedia->reverse()->take($collectionSizeLimit));
            }
        }
    }

    protected function getMediaCollection(string $collectionName): ?MediaCollection
    {
        $this->subject->registerMediaCollections();

        return collect($this->subject->mediaCollections)
            ->first(fn (MediaCollection $collection) => $collection->name === $collectionName);
    }

    /**
     * @throws FileUnacceptableForCollection
     */
    protected function guardAgainstDisallowedFileAdditions(HasMedia $hasMedia, Media $media)
    {
        $file = PendingFile::createFromMedia($media);

        if (! $collection = $this->getMediaCollection($media->collection_name)) {
            return;
        }

        if (! ($collection->acceptsFile)($file, $hasMedia)) {
            throw FileUnacceptableForCollection::create($file, $collection, $hasMedia);
        }

        if (! empty($collection->acceptsMimeTypes) && ! in_array($file->mimeType, $collection->acceptsMimeTypes)) {
            throw FileUnacceptableForCollection::create($file, $collection, $hasMedia);
        }
    }
}
