<?php

namespace Redtree\FileLibrary\Media;

use Exception;
use Illuminate\Contracts\Filesystem\Factory;
use Illuminate\Support\Facades\Storage;
use League\Flysystem\FilesystemException;
use Redtree\FileLibrary\Conversions\ConversionCollection;
use Redtree\FileLibrary\Conversions\FileManipulator;
use Redtree\FileLibrary\Exceptions\InvalidConversion;
use Redtree\FileLibrary\Media\Events\MediaHasBeenAdded;
use Redtree\FileLibrary\Media\Models\Media;
use Redtree\FileLibrary\Support\File;
use Redtree\FileLibrary\Support\PathGenerator;
use Redtree\FileLibrary\Support\RemoteFile;

class Filesystem
{
    protected Factory $filesystem;

    protected array $customRemoteHeaders = [];

    public function __construct(Factory $filesystem)
    {
        $this->filesystem = $filesystem;
    }

    public function add(string $file, Media $media, ?string $targetFileName = null): void
    {
        $this->copyToMediaLibrary($file, $media, null, $targetFileName);

        event(new MediaHasBeenAdded($media));

        app(FileManipulator::class)->createDerivedFiles($media);
    }

    /**
     * @throws FilesystemException
     */
    public function addRemote(RemoteFile $file, Media $media, ?string $targetFileName = null): void
    {
        $this->copyToMediaLibraryFromRemote($file, $media, null, $targetFileName);

        event(new MediaHasBeenAdded($media));

        app(FileManipulator::class)->createDerivedFiles($media);
    }

    public function copyToMediaLibrary(string $pathToFile, Media $media, ?string $type = null, ?string $targetFileName = null): void
    {
        $destinationFileName = $targetFileName ?: pathinfo($pathToFile, PATHINFO_BASENAME);

        $destination = $this->getMediaDirectory($media, $type) . $destinationFileName;

        $file = fopen($pathToFile, 'r');

        $diskName = $media->disk;

        $diskDriverName = $media->getDiskDriverName();

        if ($diskDriverName === 'local') {
            $this->filesystem
                ->disk($diskName)
                ->put($destination, $file);

            fclose($file);

            return;
        }

        $this->filesystem
            ->disk($diskName)
            ->put(
                $destination,
                $file,
                $this->getRemoteHeadersForFile($pathToFile, $media->getCustomHeaders()),
            );

        if (is_resource($file)) {
            fclose($file);
        }
    }

    /**
     * @throws FilesystemException
     */
    public function copyToMediaLibraryFromRemote(RemoteFile $file, Media $media, ?string $type = null, ?string $targetFileName = null): void
    {
        $destinationFileName = $targetFileName ?: $file->getFilename();
        $destination = $this->getMediaDirectory($media, $type) . $destinationFileName;
        $diskDriverName = $media->getDiskDriverName();

        if ($this->shouldCopyFileOnDisk($file, $media, $diskDriverName)) {
            $this->copyFileOnDisk($file->getKey(), $destination, $media->disk);

            return;
        }

        $storage = Storage::disk($file->getDisk());

        $headers = $diskDriverName === 'local'
            ? []
            : $this->getRemoteHeadersForFile(
                $file->getKey(),
                $media->getCustomHeaders(),
                $storage->mimeType($file->getKey())
            );

        $this->streamFileToDisk(
            $storage->getDriver()->readStream($file->getKey()),
            $destination,
            $media->disk,
            $headers
        );
    }

    public function getMediaDirectory(Media $media, ?string $type = null): string
    {
        $pathGenerator = app(PathGenerator::class);
        $directory = '';

        if ($type === null) {
            $directory = $pathGenerator->getPath($media);
        }

        if ($type === 'conversions') {
            $directory = $pathGenerator->getPathForConversions($media);
        }

        if ($media->getDiskDriverName() !== 's3') {
            $this->filesystem->disk($media->disk)->makeDirectory($directory);
        }

        return $directory;
    }

    public function addCustomRemoteHeaders(array $customRemoteHeaders): void
    {
        $this->customRemoteHeaders = $customRemoteHeaders;
    }

    public function getRemoteHeadersForFile(string $file, array $mediaCustomHeaders = [], string $mimeType = null): array
    {
        $mimeTypeHeader = ['ContentType' => $mimeType ?: File::getMimeType($file)];

        $extraHeaders = config('file-library.remote.extra_headers');

        return array_merge(
            $mimeTypeHeader,
            $extraHeaders,
            $this->customRemoteHeaders,
            $mediaCustomHeaders
        );
    }

    /**
     * @throws InvalidConversion
     */
    public function getStream(Media $media, string $conversionName = '')
    {
        if ($conversionName === '') {
            $sourceFile = $this->getMediaDirectory($media) . '/' . $media->file_name;
        } else {
            $conversion = ConversionCollection::createForMedia($media)->getByName($conversionName);

            $sourceFile = $this->getMediaDirectory($media, 'conversions') . '/' . $conversion->getConversionFile($media);
        }

        return $this->filesystem->disk($media->disk)->readStream($sourceFile);
    }

    /**
     * @throws InvalidConversion
     */
    public function copyFromMediaLibrary(Media $media, string $targetFile): string
    {
        file_put_contents($targetFile, $this->getStream($media));

        return $targetFile;
    }

    public function removeAllFiles(Media $media): void
    {
        $mediaDirectory = $this->getMediaDirectory($media);
        $conversionsDirectory = $this->getMediaDirectory($media, 'conversions');

        collect([$mediaDirectory, $conversionsDirectory])
            ->each(function (string $directory) use ($media) {
                try {
                    if ($this->filesystem->disk($media->disk)->exists($directory)) {
                        $this->filesystem->disk($media->disk)->deleteDirectory($directory);
                    }
                } catch (Exception $exception) {
                    report($exception);
                }
            });
    }

    public function removeFile(Media $media, string $path): void
    {
        $this->filesystem->disk($media->disk)->delete($path);
    }

    protected function shouldCopyFileOnDisk(RemoteFile $file, Media $media, string $diskDriverName): bool
    {
        if ($file->getDisk() !== $media->disk) {
            return false;
        }

        if ($diskDriverName === 'local') {
            return true;
        }

        if (count($media->getCustomHeaders()) > 0) {
            return false;
        }

        if (count(config('file-library.remote.extra_headers')) > 0) {
            return false;
        }

        return true;
    }

    protected function copyFileOnDisk(string $file, string $destination, string $disk): void
    {
        $this->filesystem
            ->disk($disk)
            ->copy($file, $destination);
    }

    /**
     * @throws FilesystemException
     */
    protected function streamFileToDisk($stream, string $destination, string $disk, array $headers): void
    {
        $this->filesystem
            ->disk($disk)
            ->getDriver()
            ->writeStream($destination, $stream, $headers);
    }
}
