<?php

namespace Redtree\FileLibrary\Media;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Collection;
use Redtree\FileLibrary\Exceptions\RequestDoesNotHaveFile;
use Redtree\FileLibrary\Exceptions\UnknownType;
use Redtree\FileLibrary\Support\RemoteFile;

class FileAdderFactory
{
    /**
     * @throws UnknownType
     */
    public static function create(Model $subject, string|UploadedFile|RemoteFile $file): FileAdder
    {
        /** @var FileAdder $fileAdder */
        $fileAdder = app(FileAdder::class);

        return $fileAdder
            ->setSubject($subject)
            ->setFile($file);
    }

    /**
     * @throws UnknownType
     */
    public static function createFromDisk(Model $subject, string $key, string $disk): FileAdder
    {
        return self::create($subject, new RemoteFile($key, $disk));
    }

    public static function createFromRequest(Model $subject, string $key): FileAdder
    {
        return self::createMultipleFromRequest($subject, [$key])->first();
    }

    public static function createMultipleFromRequest(Model $subject, array $keys = []): Collection
    {
        return collect($keys)
            ->map(function (string $key) use (&$subject) {
                $search = ['[', ']', '"', "'"];
                $replace = ['.', '', '', ''];

                $key = str_replace($search, $replace, $key);

                if (! request()->hasFile($key)) {
                    throw RequestDoesNotHaveFile::create($key);
                }

                $files = request()->file($key);

                if (! is_array($files)) {
                    return self::create($subject, $files);
                }

                return array_map(fn ($file) => self::create($subject, $file), $files);
            })
            ->flatten();
    }

    public static function createAllFromRequest(Model $subject): Collection
    {
        $fileKeys = array_keys(request()->allFiles());

        return self::createMultipleFromRequest($subject, $fileKeys);
    }
}
