<?php

namespace Redtree\FileLibrary\Media\Models;

use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Contracts\Support\Htmlable;
use Illuminate\Contracts\Support\Responsable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Redtree\FileLibrary\Conversions\Conversion;
use Redtree\FileLibrary\Conversions\ConversionCollection;
use Redtree\FileLibrary\Conversions\ImageGenerators\ImageGeneratorFactory;
use Redtree\FileLibrary\Exceptions\DiskDoesNotExist;
use Redtree\FileLibrary\Exceptions\FileDoesNotExist;
use Redtree\FileLibrary\Exceptions\FileIsTooBig;
use Redtree\FileLibrary\Exceptions\FileUnacceptableForCollection;
use Redtree\FileLibrary\Exceptions\InvalidConversion;
use Redtree\FileLibrary\HasMedia;
use Redtree\FileLibrary\Media\File as MediaFile;
use Redtree\FileLibrary\Media\Filesystem;
use Redtree\FileLibrary\Media\HtmlableMedia;
use Redtree\FileLibrary\Media\Models\Collections\MediaCollection;
use Redtree\FileLibrary\Media\Models\Traits\HasUuid;
use Redtree\FileLibrary\Support\File;
use Redtree\FileLibrary\Support\TemporaryDirectory;
use Redtree\FileLibrary\Support\UrlGenerator;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * @property Model $model
 * @property string $uuid
 * @property string $model_type
 * @property string $collection_name
 * @property string $name
 * @property string $file_name
 * @property string $mime_type
 * @property int $size
 * @property string $disk
 * @property array $manipulations
 * @property array $custom_properties
 * @property array $generated_conversions
 * @property string $visibility
 * @property string $extension
 */
class Media extends Model implements Responsable, Htmlable
{
    use HasUuid;

    public const VISIBILITY_PUBLIC = 'public';
    public const VISIBILITY_PRIVATE = 'private';
    public const VISIBILITY_CUSTOM = 'custom';

    public const TYPE_OTHER = 'other';

    protected $table = 'media_files';

    protected $primaryKey = 'uuid';

    public $incrementing = false;

    protected $guarded = [];

    protected $casts = [
        'manipulations' => 'array',
        'custom_properties' => 'array',
        'generated_conversions' => 'array',
    ];

    protected static function booted(): void
    {
        self::deleted(function (self $media) {
            if (in_array(SoftDeletes::class, class_uses_recursive($media))) {
                if (! $media->isForceDeleting()) {
                    return;
                }
            }

            app(Filesystem::class)->removeAllFiles($media);
        });
    }

    public function model(): MorphTo
    {
        return $this->morphTo();
    }

    public function getExtensionAttribute(): string
    {
        return pathinfo($this->file_name, PATHINFO_EXTENSION);
    }

    public function getHumanReadableSizeAttribute(): string
    {
        return File::getHumanReadableSize($this->size);
    }

    public function getTypeAttribute(): string
    {
        $type = $this->getTypeFromExtension();

        if ($type !== self::TYPE_OTHER) {
            return $type;
        }

        return $this->getTypeFromMime();
    }

    public function hasCustomProperty(string $propertyName): bool
    {
        return Arr::has($this->custom_properties, $propertyName);
    }

    public function getCustomProperty(string $propertyName, $default = null)
    {
        return Arr::get($this->custom_properties, $propertyName, $default);
    }

    public function setCustomProperty(string $name, $value): self
    {
        $customProperties = $this->custom_properties;

        Arr::set($customProperties, $name, $value);

        $this->custom_properties = $customProperties;

        return $this;
    }

    public function forgetCustomProperty(string $name): self
    {
        $customProperties = $this->custom_properties;

        Arr::forget($customProperties, $name);

        $this->custom_properties = $customProperties;

        return $this;
    }

    public function setCustomHeaders(array $customHeaders): self
    {
        $this->setCustomProperty('custom_headers', $customHeaders);

        return $this;
    }

    public function getCustomHeaders(): array
    {
        return $this->getCustomProperty('custom_headers', []);
    }

    public function getDiskDriverName(): string
    {
        return strtolower(config("filesystems.disks.$this->disk.driver"));
    }

    public function newCollection(array $models = []): MediaCollection
    {
        return new MediaCollection($models);
    }

    public function getMediaConversionNames(): array
    {
        $conversions = ConversionCollection::createForMedia($this);

        return $conversions->map(fn (Conversion $conversion) => $conversion->getName())->toArray();
    }

    public function getGeneratedConversions(): Collection
    {
        return collect($this->generated_conversions ?? []);
    }


    public function markAsConversionGenerated(string $conversionName): self
    {
        $generatedConversions = $this->generated_conversions;

        Arr::set($generatedConversions, $conversionName, true);

        $this->generated_conversions = $generatedConversions;

        $this->save();

        return $this;
    }

    public function markAsConversionNotGenerated(string $conversionName): self
    {
        $generatedConversions = $this->generated_conversions;

        Arr::set($generatedConversions, $conversionName, false);

        $this->generated_conversions = $generatedConversions;

        $this->save();

        return $this;
    }

    public function hasGeneratedConversion(string $conversionName): bool
    {
        $generatedConversions = $this->getGeneratedConversions();

        return $generatedConversions[$conversionName] ?? false;
    }

    public function getUrl(string $conversionName = ''): string
    {
        $urlGenerator = UrlGenerator::createForMedia($this, $conversionName);

        return $urlGenerator->getUrl();
    }

    public function getPath(string $conversionName = ''): string
    {
        $urlGenerator = UrlGenerator::createForMedia($this, $conversionName);

        return $urlGenerator->getPath();
    }

    public function getTypeFromExtension(): string
    {
        $imageGenerator = ImageGeneratorFactory::forExtension($this->extension);

        return $imageGenerator
            ? $imageGenerator->getType()
            : static::TYPE_OTHER;
    }

    public function getTypeFromMime(): string
    {
        $imageGenerator = ImageGeneratorFactory::forMimeType($this->mime_type);

        return $imageGenerator
            ? $imageGenerator->getType()
            : static::TYPE_OTHER;
    }

    /**
     * @throws DiskDoesNotExist
     * @throws FileDoesNotExist
     * @throws FileIsTooBig
     * @throws FileUnacceptableForCollection
     * @throws InvalidConversion
     */
    public function move(HasMedia $model, string $collectionName = 'default', string $diskName = '', string $fileName = ''): self
    {
        $newMedia = $this->copy($model, $collectionName, $diskName, $fileName);

        $this->delete();

        return $newMedia;
    }

    /**
     * @throws InvalidConversion
     * @throws DiskDoesNotExist
     * @throws FileDoesNotExist
     * @throws FileIsTooBig
     * @throws FileUnacceptableForCollection
     */
    public function copy(HasMedia $model, string $collectionName = 'default', string $diskName = '', string $fileName = ''): self
    {
        $temporaryDirectory = TemporaryDirectory::create();

        $temporaryFile = $temporaryDirectory->path('/') . DIRECTORY_SEPARATOR . $this->file_name;

        /** @var Filesystem $filesystem */
        $filesystem = app(Filesystem::class);

        $filesystem->copyFromMediaLibrary($this, $temporaryFile);

        $fileAdder = $model
            ->addMedia($temporaryFile)
            ->usingName($this->name)
            ->withCustomProperties($this->custom_properties);

        if ($fileName !== '') {
            $fileAdder->usingFileName($fileName);
        }

        $newMedia = $fileAdder
            ->toMediaCollection($collectionName, $diskName);

        $temporaryDirectory->delete();

        return $newMedia;
    }

    /**
     * @throws InvalidConversion
     */
    public function stream(string $conversionName = '')
    {
        /** @var Filesystem $filesystem */
        $filesystem = app(Filesystem::class);

        return $filesystem->getStream($this, $conversionName);
    }

    public function toResponse($request): StreamedResponse
    {
        $downloadHeaders = [
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Content-Type' => $this->mime_type,
            'Content-Length' => $this->size,
            'Content-Disposition' => 'attachment; filename="' . $this->file_name . '"',
            'Pragma' => 'public',
        ];

        return response()->stream(function () {
            $stream = $this->stream();

            fpassthru($stream);

            if (is_resource($stream)) {
                fclose($stream);
            }
        }, 200, $downloadHeaders);
    }

    public function img(string $conversionName = '', $extraAttributes = []): HtmlableMedia
    {
        return (new HtmlableMedia($this))
            ->conversion($conversionName)
            ->attributes($extraAttributes);
    }

    public function toHtml(): string
    {
        return $this->img()->toHtml();
    }

    public function __invoke(...$arguments): HtmlableMedia
    {
        return $this->img(...$arguments);
    }

    public function conversionResponse(string $conversionName): StreamedResponse
    {
        $file = MediaFile::createFromMedia($this, $conversionName);

        $downloadHeaders = [
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Content-Type' => $file->mimeType,
            'Content-Length' => $file->size,
            'Content-Disposition' => 'attachment; filename="' . $file->name . '"',
            'Pragma' => 'public',
        ];

        return response()->stream(function () use (&$conversionName) {
            $stream = $this->stream($conversionName);

            fpassthru($stream);

            if (is_resource($stream)) {
                fclose($stream);
            }
        }, 200, $downloadHeaders);
    }
}
