<?php

namespace Redtree\FileLibrary\Support;

use Redtree\FileLibrary\Conversions\Conversion;
use Redtree\FileLibrary\Conversions\ConversionCollection;
use Redtree\FileLibrary\Media\Models\Media;

class UrlGenerator
{
    protected ?Media $media;

    protected ?Conversion $conversion = null;

    public static function createForMedia(Media $media, string $conversionName = ''): self
    {
        $urlGenerator = new self();

        $urlGenerator->setMedia($media);

        if ($conversionName !== '') {
            $conversion = ConversionCollection::createForMedia($media)->getByName($conversionName);

            $urlGenerator->setConversion($conversion);
        }

        return $urlGenerator;
    }

    public function setMedia(Media $media): self
    {
        $this->media = $media;

        return $this;
    }

    public function setConversion(Conversion $conversion): self
    {
        $this->conversion = $conversion;

        return $this;
    }

    public function getUrl(): string
    {
        $conversionName = null;

        if ($this->conversion !== null) {
            $conversionName = $this->conversion->getName();
        }

        if (in_array($this->media->model->getConnectionName(), array_keys(config('file-library.connection_mapping')))) {
            return $this->versionUrl(trim(config('file-library.connection_mapping.' . $this->media->model->getConnectionName()), '/') . route('media', [$this->media, $conversionName], false));
        }

        return $this->versionUrl(route('media', [$this->media, $conversionName]));
    }

    public function getPath(): string
    {
        return $this->getPathRelativeToRoot();
    }

    protected function getPathRelativeToRoot(): string
    {
        if ($this->conversion === null) {
            return app(PathGenerator::class)->getPath($this->media) . ($this->media->file_name);
        }

        return app(PathGenerator::class)->getPathForConversions($this->media)
            . $this->conversion->getConversionFile($this->media);
    }

    protected function versionUrl(string $path = ''): string
    {
        if (! config('file-library.version_urls', false)) {
            return $path;
        }

        return "$path?v={$this->media->updated_at->timestamp}";
    }
}
