<?php

namespace IoSession;

use Carbon\Carbon;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Container\Container;
use IoSession\Models\IoSession as IoSessionModel;
use Ramsey\Uuid\Uuid;
use SessionHandlerInterface;

class IoSession implements SessionHandlerInterface
{
    /**
     * Lifetime in minutes
     *
     * @var int
     */
    private $lifetime;

    /**
     * @var Container
     */
    private $container;

    /**
     * @var IoSessionModel
     */
    private $sessionClass;

    /**
     * IoSession constructor.
     * @param  int  $lifetime
     * @param  Container|null  $container
     * @throws BindingResolutionException
     */
    public function __construct(int $lifetime, Container $container = null)
    {
        $session = config('io-session.sessionClass');
        $sessionClass = new $session();
        if(! $sessionClass instanceof \IoSession\Models\IoSession) {
            throw new BindingResolutionException('Session Class must implement IoSession model.');
        }

        $this->sessionClass = $sessionClass;
        $this->container = $container;
        $this->lifetime = $lifetime;

    }

    /**
     * {@inheritdoc}
     */
    public function close()
    {
        true;
    }

    /**
     * {@inheritdoc}
     */
    public function destroy($sessionId)
    {
        $this->sessionClass->newQuery()
            ->bySessionId($sessionId)
            ->delete();
    }

    /**
     * {@inheritdoc}
     */
    public function gc($maxlifetime)
    {
        $timestamp = Carbon::now()->subSeconds($maxlifetime);
        $this->sessionClass->newQuery()
            ->where('last_activity', '<', $timestamp)
            ->delete();
    }

    /**
     * {@inheritdoc}
     */
    public function open($save_path, $name)
    {
        true;
    }

    /**
     * {@inheritdoc}
     */
    public function read($sessionId)
    {
        $session = $this->sessionClass->newQuery()
            ->bySessionId($sessionId)
            ->first();

        if (is_null($session)) {
            return '';
        }

        if ($this->isExpired($session)) {
            $this->destroy($sessionId);

            return '';
        }

        $session->update([
            'last_activity' => Carbon::now()
        ]);

        return base64_decode($session->payload);
    }

    /**
     * {@inheritdoc}
     */
    public function write($sessionId, $sessionData)
    {
        $payload = $this->getDefaultPayload($sessionData);

        $session = $this->sessionClass->newQuery()->bySessionId($sessionId)
            ->first() ?? $this->sessionClass->newInstance();

        $session->fill(array_merge($payload, [
            'app_name' => config('io-session.appName'),
            'session_id' => $sessionId,
        ]));

        if ($session->exists === false) {
            $session->uuid = Uuid::uuid4()->toString();
        }

        return $session->save();
    }

    protected function getDefaultPayload(string $sessionData):array
    {
        $payload = [
            'payload' => base64_encode($sessionData),
            'last_activity' => Carbon::now(),
        ];

        if ($this->container) {
            $this->addRequestData($payload)
                ->addUserData($payload);
        }

        return $payload;
    }

    protected function addRequestData(array &$payload):self
    {
        $payload = array_merge($payload, [
            'user_agent' => $this->userAgent(),
            'ip_address' => $this->ipAddress(),
        ]);

        return $this;
    }

    protected function addUserData(array &$payload):self
    {
        $payload['user_id'] = $this->userId();

        return $this;
    }

    protected function isExpired(IoSessionModel $session):bool
    {
        return $session
            ->last_activity
            ->lt(Carbon::now()->subMinutes($this->lifetime));
    }

    /**
     * Get the currently authenticated user's ID.
     *
     * @return string|integer|null
     * @throws BindingResolutionException
     */
    protected function userId()
    {
        return $this->container->make(Guard::class)->id();
    }

    /**
     * Get the user agent for the current request.
     */
    protected function userAgent():string
    {
        $userAgent = (string) $this->container
            ->make('request')
            ->header('User-Agent');

        return substr($userAgent, 0, 500);
    }

    /**
     * Get the IP address for the current request.
     *
     * @return string
     * @throws BindingResolutionException
     */
    protected function ipAddress()
    {
        return $this->container->make('request')->ip();
    }
}
