<?php

namespace io\Settings;

use ArrayIterator;
use Countable;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use io\Settings\Model\Setting;
use io\Settings\Settings\Elements\Tab;
use IteratorAggregate;

class Settings implements Arrayable, IteratorAggregate, Countable
{
    /**
     * @var Collection
     */
    private $settings;

    /**
     * Create a new Settings Object
     *
     * @return void
     */
    public function __construct()
    {
        $this->settings = new Collection();
    }

    /**
     * Add a new Tab to the Settings
     *
     * @param Tab $tab new Tab Object
     *
     * @return self
     */
    public function addTab(Tab $tab)
    {
        $this->settings->put($tab->getKey(), $tab);

        return $this;
    }

    /**
     * Get a Tab by name
     *
     * @var    string $key
     * @return boolean
     */
    public function removeTab($key):bool
    {
        $this->settings->forget($key);

        return ! $this->settings->has($key);
    }

    /**
     * Get a Tab by name
     *
     * @var    string $key
     * @return Tab|null
     */
    public function getTab($key):?Tab
    {
        return $this->settings->get(kebab_case($key));
    }

    /**
     * Save settings and assign to Model
     *
     * @param Model $model
     * @return bool
     */
    public function create(Model $model):bool
    {
        $this->settings->each(function (Tab $tab) use (&$model) {
            $setting = [
                'key' => $tab->getKey(),
                'settings' => $tab->toArray(),
                'hashsum' => hash('sha256', $tab->toJson()),
                'setable_type' => \get_class($model),
            ];

            Setting::firstOrCreate([
                'key' => $tab->getKey(),
                'setable_type' => \get_class($model),
            ], $setting);
        });

        return true;
    }

    /**
     * @return array
     */
    public function toArray():array
    {
        return $this->settings->toArray();
    }

    /**
     * Get an iterator for the items.
     *
     * @return ArrayIterator
     */
    public function getIterator():ArrayIterator
    {
        return new ArrayIterator($this->settings);
    }

    /**
     * @return int
     */
    public function count():int
    {
        return $this->settings->count();
    }
}
