<?php

namespace io\Settings\Settings\Elements;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;

abstract class Field implements Arrayable, Jsonable
{
    protected $attributes = [];

    /**
     * Create a new Field
     *
     * @param  string $name form name of the settingsfield
     * @return void
     */
    public function __construct($name)
    {
        $this->set('name', $this->fieldcase($name));
    }

    /**
     * Set the labeltext
     *
     * @var    string $text
     * @return this
     */
    public function label($text)
    {
        $this->set('label', $text);

        return $this;
    }

    /**
     * @param  mixed $element
     * @return this
     */
    public function element($element = 'input')
    {
        $this->set('element', $element);

        return $this;
    }

    /**
     * @param  string $type
     * @return this
     */
    public function type($type = 'text')
    {
        $this->set('type', $type);

        return $this;
    }

    /**
     * Set custom field attribute
     *
     * @param string $name
     * @param mixed  $value
     */
    public function setDataAttribute($name, $value)
    {
        $this->set('data-'.$this->fieldcase($name), $value);

        return $this;
    }

    /**
     * Set custom field attribute
     *
     * @param string $name
     * @param mixed  $value
     */
    public function setCustomAttribute($name, $value)
    {
        $this->set($this->fieldcase($name), $value);

        return $this;
    }

    /**
     * Set field required attribute
     *
     * @param  bool $value
     * @return this
     */
    public function required(bool $required)
    {
        $this->set('required', $required);

        return $this;
    }

    /**
     * Set field disabled attribute
     *
     * @param  bool $value
     * @return this
     */
    public function disabled(bool $disabled)
    {
        $this->set('disabled', $disabled);

        return $this;
    }

    /**
     * Set field readonly attribute
     *
     * @param  bool $value
     * @return this
     */
    public function readonly(bool $readonly)
    {
        $this->set('readonly', $readonly);

        return $this;
    }

    /**
     * Get Field attributes
     *
     * @param  string $parameters
     * @return mixed
     */
    public function get($key, $default = null)
    {
        return array_get($this->attributes, $key, $default);
    }

    /**
     * Set Laravel Validation Rules
     *
     * @param  array $array Array of laravel Rules
     * @return this
     */
    public function validationRules(array $array)
    {
        $this->set('rules', $array);

        return $this;
    }

    /**
     * @return array
     */
    public function toArray():array
    {
        return $this->attributes;
    }

    /**
     * Convert the object to its JSON representation.
     *
     * @param  int  $options
     * @return string
     */
    public function toJson($options = 0):string
    {
        return json_encode($this->toArray(), $options);
    }

    /**
     * Set Field attributes
     *
     * @param  mixed $parameters
     * @return void
     */
    protected function set($key, $value)
    {
        $this->attributes[$key] = $value;
    }

    /**
     *
     * Remove Field attribute
     * @param  mixed $key
     * @return void
     */
    protected function remove($key)
    {
        unset($this->attributes[$key]);
    }

    /**
     * @param  string $string
     * @return string
     */
    protected function fieldcase($string):string
    {
        return str_replace(' ', '-', $string);
    }
}
