<?php

namespace io\Settings\Settings\Elements;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;

abstract class Field implements Arrayable, Jsonable
{
    protected $attributes = [];

    /**
     * Create a new Field
     *
     * @param  string $name form name of the settingsfield
     * @return void
     */
    public function __construct($name)
    {
        $this->set('name', $this->fieldcase($name));
    }

    /**
     * Set the labeltext
     *
     *
     * @return Field
     * @var    string $text
     */
    public function label($text)
    {
        $this->set('label', $text);

        return $this;
    }

    /**
     * @param  mixed  $element
     * @return Field
     */
    public function element($element = 'input')
    {
        $this->set('element', $element);

        return $this;
    }

    /**
     * @param  string  $type
     * @return Field
     */
    public function type($type = 'text')
    {
        $this->set('type', $type);

        return $this;
    }

    /**
     * Set custom field attribute
     *
     * @param  string  $name
     * @param  mixed  $value
     * @return Field
     */
    public function setDataAttribute($name, $value)
    {
        $this->set('data-'.$this->fieldcase($name), $value);

        return $this;
    }

    /**
     * Set custom field attribute
     *
     * @param  string  $name
     * @param  mixed  $value
     * @return Field
     */
    public function setCustomAttribute($name, $value)
    {
        $this->set($this->fieldcase($name), $value);

        return $this;
    }

    /**
     * Set field required attribute
     *
     * @param  bool  $required
     * @return Field
     */
    public function required(bool $required)
    {
        $this->set('required', $required);

        return $this;
    }

    /**
     * Set field disabled attribute
     *
     * @param  bool  $disabled
     * @return Field
     */
    public function disabled(bool $disabled)
    {
        $this->set('disabled', $disabled);

        return $this;
    }

    /**
     * Set field readonly attribute
     *
     * @param  bool  $readonly
     * @return Field
     */
    public function readonly(bool $readonly)
    {
        $this->set('readonly', $readonly);

        return $this;
    }

    /**
     * Get Field attributes
     *
     * @param $key
     * @param  null  $default
     * @return mixed
     */
    public function get($key, $default = null)
    {
        return array_get($this->attributes, $key, $default);
    }

    /**
     * Set Laravel Validation Rules
     *
     * @param  array  $array  Array of laravel Rules
     * @return Field
     */
    public function validationRules(array $array)
    {
        $this->set('rules', $array);

        return $this;
    }

    /**
     * @return array
     */
    public function toArray():array
    {
        return $this->attributes;
    }

    /**
     * Convert the object to its JSON representation.
     *
     * @param  int $options
     * @return string
     */
    public function toJson($options = 0):string
    {
        return json_encode($this->toArray(), $options);
    }

    /**
     * Set Field attributes
     *
     * @param $key
     * @param $value
     * @return void
     */
    protected function set($key, $value)
    {
        $this->attributes[$key] = $value;
    }

    /**
     * Remove Field attribute
     *
     * @param  mixed $key
     * @return void
     */
    protected function remove($key)
    {
        unset($this->attributes[$key]);
    }

    /**
     * @param  string $string
     * @return string
     */
    protected function fieldcase($string):string
    {
        return str_replace(' ', '-', $string);
    }
}
