<?php

namespace io\Settings\Http\Controller;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Illuminate\Support\Fluent;
use io\Settings\Exceptions\ModelNotFoundException;
use io\Settings\Routes\Routes;

class SettingsController extends Controller
{
    /**
     * Show the form for creating a new resource.
     *
     * @param $modelShort
     * @param $id
     * @return Response
     * @throws ModelNotFoundException
     */
    public function create($modelShort, $id)
    {
        $modelClass = array_search($modelShort, config('ioSettings.morphMap'));
        if($modelClass === false){
            throw new ModelNotFoundException('No mapping in config found for "'.$modelShort.'"');
        }
        $model = with(new $modelClass())->resolveRouteBinding($id);

        if(!$model){
            throw new ModelNotFoundException('No Entry found for "'.$id.'" of '.$modelClass);
        }

        $defaultSettings = $model->settings()->with(
            ['values' => function ($query) use (&$model) {
                $query->where('model_type', get_class($model))
                    ->where('model_id', $model->id);
            }]
        )->get();

        $route = Routes::routeForModel($model, 'store');
        $settings = collect();
        $defaultSettings->sort(function($setting){
            return Arr::get($setting->settings, 'position', $setting->key);
        })->each(function ($setting) use (&$settings) {
            $fluent = new Fluent([
                'key' => $setting->key,
                'id' => $setting->id,
                'label' => Arr::get($setting->settings, 'label',$setting->key),
            ]);
            $fields = [];
            $fieldsSorted = collect($setting->settings['fields'])->sortBy(function($field){
                return (int) Arr::get($field,'position',0);
            });
            foreach ($fieldsSorted as $key => $attributes) {
                $value = $setting->values->firstWhere('name', $key);
                if ($value) {
                    if (count($value->values) === 1) {
                        $attributes = array_merge($attributes, ['old' => Arr::first($value->values)]);
                    } else {
                        $attributes = array_merge($attributes, ['old' => $value->values]);
                    }
                }
                $fields[$key] = $attributes;
            }
            $fluent->fields = $fields;
            $settings->push($fluent);
        });

        return view(config('ioSettings.view'), compact('settings', 'route'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  Request  $request
     * @param $modelShort
     * @param $id
     * @return Response
     * @throws ModelNotFoundException
     */
    public function store(Request $request, $modelShort, $id)
    {
        $modelClass = array_search($modelShort, config('ioSettings.morphMap'));
        if($modelClass === false){
            throw new ModelNotFoundException('No mapping in config found for "'.$modelShort.'"');
        }
        $model = with(new $modelClass())->resolveRouteBinding($id);

        if(!$model){
            throw new ModelNotFoundException('No Entry found for "'.$id.'" of '.$modelClass);
        }

        $settings = $model->settings()->find($request->get('setting_id'));

        if (! $settings) {
            throw new ModelNotFoundException();
        }

        $rules = [];
        foreach ($settings->settings['fields'] as $name => $attributes) {
            array_merge($rules, [$name => array_get($attributes, 'rules', [])]);
        }


        $validator = validator($request->all(), $rules);
        if ($validator->fails()) {
            return back(422)->withErrors($validator->errors);
        }

        foreach (Arr::get($settings->settings, 'fields', []) as $field => $attributes) {
            if (Arr::get($field, 'element') === 'select' ||
                in_array(Arr::get($field, 'type'), ['checkbox', 'radio'])) {
                $value = $request->get($field);
            } else {
                $value = [$request->get($field, array_get($attributes, 'value'))];
            }
            if (! $value) {
                continue;
            }
            $model->settingsValues()->updateOrCreate(
                [
                    'setting_id' => $settings->id,
                    'key' => $settings->key,
                    'name' => $field,
                ], [
                    'setting_id' => $settings->id,
                    'key' => $settings->key,
                    'name' => $field,
                    'values' => $value
                ]
            );
        }

        return back()->with('success','Saved');
    }
}
