<?php

namespace io\Settings\Commands;

use Exception;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use io\Settings\Contracts\SettingsCreator;
use io\Settings\Exceptions\TypeException;
use io\Settings\Settings;
use Symfony\Component\Console\Exception\LogicException;

class CreateSettingsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'settings:create {--force} {--no-update}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Save settings to defined models.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws TypeException
     */
    public function handle()
    {
        $configs = $this->getConfig();


        foreach ($configs as $class => $model) {
            $this->createSettings($class, $model);
        }
    }

    private function createSettings($creatorClass, $model)
    {
        $this->checkModel($model);
        $class = new $creatorClass();

        if (! $class instanceof SettingsCreator) {
            throw new TypeException($creatorClass .' needs to implement '. SettingsCreator::class);
        }

        /**
         * @var $class SettingsCreator
         */
        $settings = with(new $class())->generate();
        /**
         * @var $settings Settings
         */
        try{
            $settings->createOrUpdateFromConsole(new $model(), $this);
        }catch (Exception $e) {
            $this->error('Something went wrong while create and update the settings');
            $this->error($e->getMessage());
        }
    }

    /**
     * Check if Model Parameter is valid.
     *
     * @param  string $model
     * @throws ModelNotFoundException
     * @throws LogicException
     * @return void
     */
    private function checkModel($model)
    {
        if (empty($model)) {
            throw new LogicException('Model needs to be set.');
        }

        if (! class_exists($model)) {
            throw new ModelNotFoundException('Class '.$model.' not found.');
        }

        $modelClass = new $model();
        if (! $modelClass instanceof Model) {
            throw new LogicException('Model needs instance of '.Model::class);
        }
        unset($modelClass);
    }

    /**
     * Get Settings from Config Class
     *
     * @return array
     * @throws Exception
     */
    private function getConfig():array
    {
        $configClasses = config('ioSettings.generatorClasses');

        if (empty($configClasses)) {
            throw new Exception('Configuration class needs to be set in config file.');
        }

        return $configClasses;
    }
}
