<?php

namespace io\Settings;

use ArrayIterator;
use Countable;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use io\Settings\Model\Setting;
use io\Settings\Model\SettingsValue;
use io\Settings\Settings\Elements\Tab;
use IteratorAggregate;

class Settings implements Arrayable, IteratorAggregate, Countable
{
    /**
     * @var Collection
     */
    private $settings;

    /**
     * Create a new Settings Object
     *
     * @return void
     */
    public function __construct()
    {
        $this->settings = new Collection();
    }

    /**
     * Add a new Tab to the Settings
     *
     * @param Tab $tab new Tab Object
     *
     * @return self
     */
    public function addTab(Tab $tab)
    {
        $this->settings->put($tab->getKey(), $tab);

        return $this;
    }

    /**
     * Get a Tab by name
     *
     * @var    string $key
     * @return boolean
     */
    public function removeTab($key):bool
    {
        $this->settings->forget($key);

        return ! $this->settings->has($key);
    }

    /**
     * Get a Tab by name
     *
     * @var    string $key
     * @return Tab|null
     */
    public function getTab($key):?Tab
    {
        return $this->settings->get(kebab_case($key));
    }

    /**
     * Save or Update settings and assign to Model
     *
     * @param  Model  $model
     * @param  bool  $update
     * @return bool
     * @throws Exception
     */
    public function createOrUpdate(Model $model, bool $update = true):bool
    {
        if(app()->runningInConsole()) {
            throw new Exception('Running in Console. Use createOrUpdateFromConsole for console mode.');
        }
        $this->settings->each(
            function (Tab $tab) use (&$model, &$update) {
                $hashsum = hash('sha256', $tab->toJson());
                $setting = $this->save($model,$tab,$hashsum);

                if(! $setting->wasRecentlyCreated && $setting->hashsum !== $hashsum && $update === true) {
                    $this->update($setting, $tab, $hashsum);
                }

            }
        );
        $this->cleanUpOld($model);

        return true;
    }

    /**
     * Save settings and assign to Model
     *
     * @param  Model  $model
     * @param  Command  $cmd
     * @return bool
     * @throws Exception
     */
    public function createOrUpdateFromConsole(Model $model, Command $cmd):bool
    {
        if(! app()->runningInConsole()) {
            throw new Exception('Not running in Console. Use createOrUpdate for default mode.');
        }
        $this->settings->each(
            function (Tab $tab) use (&$model, &$cmd) {
                $hashsum = hash('sha256', $tab->toJson());
                $setting = $this->save($model,$tab,$hashsum);
                if($cmd->option('no-update')){
                    return;
                }
                if(! $setting->wasRecentlyCreated && $setting->hashsum !== $hashsum) {
                    $question = sprintf('The following setting will be updated. Continue? '.PHP_EOL.'"%s":"%s"'.PHP_EOL, get_class($model),$tab->getKey());
                    if($cmd->option('force') || $cmd->confirm($question)) {
                        $this->update($setting, $tab, $hashsum);
                    }
                }
            }
        );

        $this->cleanUpOld($model);

        return true;
    }

    /**
     * @return array
     */
    public function toArray():array
    {
        return $this->settings->toArray();
    }

    /**
     * Get an iterator for the items.
     *
     * @return ArrayIterator
     */
    public function getIterator():ArrayIterator
    {
        return new ArrayIterator($this->settings);
    }

    /**
     * @return int
     */
    public function count():int
    {
        return $this->settings->count();
    }

    /**
     * @param  Model  $model
     * @param  Tab  $tab
     * @param  string $hashsum
     * @return Setting
     */
    private function save(Model $model, Tab $tab,string  $hashsum):Setting
    {
        $setting = [
            'key' => $tab->getKey(),
            'settings' => $tab->toArray(),
            'hashsum' => $hashsum,
            'setable_type' => get_class($model),
        ];

        $setting = Setting::firstOrCreate(
            [
                'key' => $tab->getKey(),
                'setable_type' => \get_class($model),
            ], $setting
        );

        return $setting;
    }

    /**
     * @param  Setting  $setting
     * @param  Tab  $tab
     * @param  string $hashsum
     */
    private function update(Setting $setting, Tab $tab, string $hashsum):void
    {
        $setting->update([
            'settings' => $tab->toArray(),
            'hashsum' => $hashsum,
        ]);
    }


    /**
     * @param  Model  $model
     */
    protected function cleanUpOld(Model $model):void
    {
        $tabKeys = [];
        $this->settings->each(function (Tab $tab) use(&$tabKeys){
            $tabKeys[] = $tab->getKey();
        });
        $modelClass = get_class($model);

        SettingsValue::where('model_type',$modelClass)
            ->whereNotIn('key', $tabKeys)
            ->delete();

        Setting::where('setable_type',$modelClass)
            ->whereNotIn('key', $tabKeys)
            ->delete();
    }
}
