<?php

namespace io\Settings\Settings\Elements;

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;

class Tab implements Arrayable, Jsonable
{

    /**
     * @var string
     */
    private $key;

    /**
     * @var Collection
     */
    private $fields;

    /**
     * @var array
     */
    protected $attributes = [];

    /**
     * Create a new Settings Tab
     *
     * @param  string $key
     * @return void
     */
    public function __construct($key)
    {
        $this->fields = new Collection();
        $this->key = kebab_case($key);
    }

    /**
     * @param  Field  $field
     * @return self
     */
    public function addField(Field $field):self
    {
        if(!$field->get('position')){
            $field->position($this->fields->count());
        }

        $this->fields->put($field->get('name'), $field);

        return $this;
    }

    /**
     * @param  int  $position
     * @return self
     */
    public function setPosition(int $position):self
    {
        $this->attributes['position'] = $position;

        return $this;
    }

    /**
     * @return  int
     */
    public function getPosition():int
    {
        return (int) Arr::get($this->attributes,'position',0);
    }

    /**
     * @param  string  $label
     * @return self
     */
    public function setLabel(string $label):self
    {
        Arr::set($this->attributes,'label',(string) $label);

        return $this;
    }

    /**
     * @return string|null
     */
    public function getLabel():?string
    {
        return Arr::get($this->attributes,'label');
    }

    /**
     * @return string Returns the tab key
     */
    public function getKey()
    {
        return $this->key;
    }

    /**
     * @return Collection
     */
    public function fields():Collection
    {
        return $this->fields;
    }

    /**
     * return Object as Array
     *
     * @return array
     */
    public function toArray():array
    {
        return [
            'key' => $this->key,
            'label' => $this->getLabel(),
            'position' => $this->getPosition(),
            'fields' => $this->fields->toArray(),
        ];
    }

    /**
     * Convert the object to its JSON representation.
     *
     * @param  int $options
     * @return string
     */
    public function toJson($options = 0):string
    {
        return json_encode($this->toArray(), $options);
    }
}
