<?php

namespace Redtree\Tenancy\Models\System;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Redtree\FileLibrary\HasMedia;
use Redtree\FileLibrary\InteractsWithMedia;
use Redtree\Tenancy\DTO\CompanyDetailsDTO;
use Redtree\Tenancy\Services\TenantService;
use Redtree\Tenancy\Traits\HasUuid;
use Redtree\Tenancy\Traits\UsesSystemConnection;

/**
 * @property CompanyDetailsDTO $company_details
 */
class Tenant extends Model implements HasMedia
{
    use UsesSystemConnection;
    use HasUuid;
    use InteractsWithMedia;

    protected $fillable = [
        'uuid',
        'display_name',
        'is_active',
        'company_details',
    ];

    protected $casts = [
        'is_active' => 'bool',
        'company_details' => CompanyDetailsDTO::class,
    ];

    protected ?int $currentAppId = null;
    protected ?string $currentAppUrl = null;

    /**
     * @return BelongsTo|Tenant
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class);
    }

    /**
     * @return HasMany|Tenant
     */
    public function children(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    /**
     * @return BelongsToMany|App
     */
    public function apps(): BelongsToMany
    {
        return $this->belongsToMany(App::class)
            ->using(AppTenantPivot::class)
            ->withPivot(['url']);
    }

    public function scopeActive(Builder $builder): Builder
    {
        return $builder->where('is_active', true);
    }

    public function getDatabaseUuidAttribute(): string
    {
        return str_replace('-', null, $this->uuid);
    }

    public function displayName(): string
    {
        $name = trim(sprintf('%s %s', $this->company_details->name, $this->company_details->legalForm));

        if (empty($name)) {
            $name = $this->display_name;
        }

        return $name;
    }

    public function switchTo(): void
    {
        app(TenantService::class)->switchTo($this);
    }

    public function getApp(string $appName): ?App
    {
        return cache()->remember(sprintf('getApp.tenant.%s.app.%s', $this->uuid, $appName), now()->addHour(), function () use (&$appName) {
            return $this
                ->apps()
                ->where('is_io_app', true)
                ->where('io_name', $appName)
                ->first();
        });
    }

    public function getCurrentAppId(): ?int
    {
        if ($this->currentAppId === null) {
            $this->currentAppId = $this->getApp(config('tenancy.app_name'))->id;
        }

        return $this->currentAppId;
    }

    public function getAppUrl(string $appName, string $path = null): ?string
    {
        $app = $this->getApp($appName);

        if ($app === null) {
            return null;
        }

        $url = $app->pivot->url;

        if ($url === null) {
            return null;
        }

        $url = base_url($url);

        if ($path !== null) {
            if (! Str::endsWith($url, '/')) {
                $url .= '/';
            }

            $url .= ltrim($path, '/');
        }

        return $url;
    }

    public function getCurrentAppUrl(): ?string
    {
        if ($this->currentAppUrl === null) {
            $this->currentAppUrl = $this->getAppUrl(config('tenancy.app_name'));
        }

        return $this->currentAppUrl;
    }
}
