<?php

namespace Redtree\Tenancy\Models\System;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Redtree\Tenancy\DTO\UserProfileDTO;
use Redtree\Tenancy\Exceptions\PermissionDoesNotExist;
use Redtree\Tenancy\Models\Tenant\Permission;
use Redtree\Tenancy\Models\Tenant\Role;
use Redtree\Tenancy\Traits\HasUuid;
use Redtree\Tenancy\Traits\UsesSystemConnection;

/**
 * @property UserProfileDTO $profile
 */
class User extends Authenticatable implements MustVerifyEmail
{
    use UsesSystemConnection;
    use HasUuid;
    use Notifiable;

    protected $fillable = [
        'uuid',
        'email',
        'password',
        'email_verified_at',
    ];

    protected $casts = [
        'is_logged_in' => 'bool',
        'profile' => UserProfileDTO::class,
        'email_verified_at' => 'datetime',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * @return BelongsToMany|App
     */
    public function apps(): BelongsToMany
    {
        return $this->belongsToMany(App::class)
            ->using(AppUserPivot::class)
            ->withPivot(['tenant_id']);
    }

    /**
     * @return BelongsToMany|Role
     */
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class);
    }

    /**
     * @return BelongsToMany
     */
    public function groups(): BelongsToMany
    {
        return $this->belongsToMany(Group::class);
    }

    public function scopeWithRole(Builder $builder, $roles): Builder
    {
        $roles = Arr::wrap($roles);

        return $builder
            ->whereHas('roles', function (Builder $builder) use (&$roles) {
                $builder->whereIn('name', $roles);
            });
    }

    public function getDisplayNameAbbrAttribute(): string
    {
        return $this->displayNameShort();
    }

    public function displayName(): string
    {
        $name = trim(sprintf('%s %s', $this->profile->firstName, $this->profile->lastName));

        if (empty($name)) {
            return $this->email;
        }

        return $name;
    }

    public function displayNameShort(): string
    {
        if ($this->profile->firstName === null || $this->profile->lastName === null) {
            return Str::of($this->email)->substr(0, 2)->upper();
        }

        return Str::of($this->profile->firstName)->substr(0, 1)->upper() . Str::of($this->profile->lastName)->substr(0, 1)->upper();
    }

    /**
     * @param  string|array|Role  $roles
     * @param  Tenant|null  $tenant
     * @return $this
     */
    public function assignRole($roles, Tenant $tenant = null): self
    {
        $roles = collect($roles)
            ->flatten()
            ->map(function ($role) {
                if (empty($role)) {
                    return false;
                }

                return $this->getStoredRole($role);
            })
            ->filter(static function ($role) {
                return $role instanceof Role;
            })
            ->map->id
            ->all();

        if ($tenant !== null) {
            $withPivot = [];

            foreach ($roles as $role) {
                $withPivot[$role] = $tenant->id;
            }

            $roles = $withPivot;
        }

        $this
            ->roles()
            ->syncWithoutDetaching($roles);

        $this->load('roles');

        return $this;
    }

    /**
     * @param  string|Role  $role
     * @return $this
     */
    public function removeRole($role): self
    {
        $this
            ->roles()
            ->detach($this->getStoredRole($role));

        $this->load('roles');

        return $this;
    }

    /**
     * @param  array|string|Role  ...$roles
     * @return $this
     */
    public function syncRoles(...$roles): self
    {
        $this
            ->roles()
            ->detach();

        return $this->assignRole($roles);
    }

    /**
     * @param  string|int|array|Role|Collection  $roles
     * @return bool
     */
    public function hasRole($roles): bool
    {
        $userRoles = $this->roles;

        if (is_string($roles)) {
            return $userRoles->contains('name', $roles);
        }

        if (is_int($roles)) {
            return $userRoles->contains('id', $roles);
        }

        if ($roles instanceof Role) {
            return $userRoles->contains('id', $roles->id);
        }

        if (is_array($roles)) {
            foreach ($roles as $role) {
                if ($this->hasRole($role)) {
                    return true;
                }
            }

            return false;
        }

        return $roles
            ->intersect($userRoles)
            ->isNotEmpty();
    }

    /**
     * @param  string|int|Permission  $permission
     * @return bool
     */
    public function hasPermissionTo($permission): bool
    {
        if (is_string($permission)) {
            $permission = Permission::findByName($permission);
        }

        if (is_int($permission)) {
            $permission = Permission::findById($permission);
        }

        if (! $permission instanceof Permission) {
            throw new PermissionDoesNotExist();
        }

        return $this->hasPermissionViaRole($permission) || $this->hasPermissionViaGroup($permission);
    }

    public function checkPermissionTo($permission): bool
    {
        try {
            return $this->hasPermissionTo($permission);
        } catch (PermissionDoesNotExist $e) {
            return false;
        }
    }

    protected function hasPermissionViaRole(Permission $permission): bool
    {
        return $this->hasRole($permission->roles);
    }

    protected function hasPermissionViaGroup(Permission $permission): bool
    {
        $this->groups->loadMissing('roles');

        return $permission
            ->roles
            ->intersect($this->groups
                ->pluck('roles')
                ->flatten())
            ->isNotEmpty();
    }

    protected function getStoredRole($role): Role
    {
        if (is_numeric($role)) {
            return Role::findById($role);
        }

        if (is_string($role)) {
            return Role::findByName($role);
        }

        return $role;
    }
}
