<?php

namespace Redtree\Tenancy\Models\Tenant;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Redtree\Tenancy\Exceptions\PermissionAlreadyExists;
use Redtree\Tenancy\Exceptions\PermissionDoesNotExist;
use Redtree\Tenancy\Services\PermissionsCache;
use Redtree\Tenancy\Traits\RefreshesPermissionCache;
use Redtree\Tenancy\Traits\UsesTenantConnection;
use Redtree\Tenancy\Traits\WithAppName;

class Permission extends Model
{
    use UsesTenantConnection;
    use RefreshesPermissionCache;
    use WithAppName;

    protected $fillable = [
        'name',
    ];

    /**
     * @return BelongsToMany|Role
     */
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class);
    }

    public static function create(array $attributes = [])
    {
        if (static::where('name', $attributes['name'])->first()) {
            throw PermissionAlreadyExists::create($attributes['name']);
        }

        return static::query()
            ->create($attributes);
    }

    public static function findOrCreate(string $name): self
    {
        $permission = static::getPermissions(['name' => $name])
            ->first();

        if ($permission === null) {
            $permission = static::query()
                ->create([
                    'name' => $name,
                ]);
        }

        return $permission;
    }

    public static function findByName(string $name): self
    {
        $permission = static::getPermissions(['name' => $name])
            ->first();

        if ($permission === null) {
            throw PermissionDoesNotExist::named($name);
        }

        return $permission;
    }

    public static function findById(int $id): self
    {
        $permission = static::getPermissions(['id' => $id])
            ->first();

        if (! $permission) {
            throw PermissionDoesNotExist::withId($id);
        }

        return $permission;
    }

    protected static function getPermissions(array $params = []): Collection
    {
        return app(PermissionsCache::class)->getPermissions($params);
    }
}
