<?php

namespace Redtree\Tenancy\Models\Tenant;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Redtree\Tenancy\Exceptions\RoleAlreadyExists;
use Redtree\Tenancy\Exceptions\RoleDoesNotExist;
use Redtree\Tenancy\Models\System\Group;
use Redtree\Tenancy\Models\System\User;
use Redtree\Tenancy\Services\PermissionsCache;
use Redtree\Tenancy\Traits\RefreshesPermissionCache;
use Redtree\Tenancy\Traits\UsesTenantConnection;
use Redtree\Tenancy\Traits\WithAppName;

class Role extends Model
{
    use UsesTenantConnection;
    use RefreshesPermissionCache;
    use WithAppName;

    protected $fillable = [
        'name',
        'is_tenant',
    ];

    protected $casts = [
        'is_tenant' => 'bool',
    ];

    /**
     * @return BelongsToMany|Permission
     */
    public function permissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class);
    }

    /**
     * @return BelongsToMany|User
     */
    public function users(): BelongsToMany
    {
        return $this
            ->belongsToMany(User::class)
            ->withPivot('tenant_id');
    }

    /**
     * @return BelongsToMany|Group
     */
    public function groups(): BelongsToMany
    {
        return $this->belongsToMany(Group::class);
    }

    public static function create(array $attributes = [])
    {
        if (static::where('name', $attributes['name'])->first()) {
            throw RoleAlreadyExists::create($attributes['name']);
        }

        return static::query()
            ->create($attributes);
    }

    public static function findOrCreate(string $name): Role
    {
        $role = static::where('name', $name)->first();

        if ($role === null) {
            $role = static::query()
                ->create([
                    'name' => $name,
                ]);
        }

        return $role;
    }

    public static function findByName(string $name): Role
    {
        $role = static::where('name', $name)->first();

        if ($role === null) {
            throw RoleDoesNotExist::named($name);
        }

        return $role;
    }

    public static function findById(int $id): Role
    {
        $role = static::where('id', $id)->first();

        if (! $role) {
            throw RoleDoesNotExist::withId($id);
        }

        return $role;
    }

    /**
     * @param  string|array|Permission|Collection  ...$permissions
     * @return Role
     */
    public function givePermissionTo(...$permissions): Role
    {
        $permissions = collect($permissions)
            ->flatten()
            ->map(function ($permission) {
                if (empty($permission)) {
                    return false;
                }

                return $this->getStoredPermission($permission);
            })
            ->filter(function ($permission) {
                return $permission instanceof Permission;
            })
            ->map->id
            ->all();

        $this
            ->permissions()
            ->syncWithoutDetaching($permissions);

        $this->load('permissions');

        app(PermissionsCache::class)->forgetCachedPermissions();

        return $this;
    }

    /**
     * @param  string|array|Permission|Collection  $permissions
     * @return Permission|Permission[]|Collection
     */
    protected function getStoredPermission($permissions)
    {
        if (is_numeric($permissions)) {
            return Permission::findById($permissions);
        }

        if (is_string($permissions)) {
            return Permission::findByName($permissions);
        }

        if (is_array($permissions)) {
            return Permission::whereIn('name', $permissions)
                ->get();
        }

        return $permissions;
    }
}
