<?php

namespace Redtree\Tenancy\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\URL;
use Redtree\Tenancy\DTO\MailSmtpDTO;
use Redtree\Tenancy\Events\TenantSwitched;
use Redtree\Tenancy\Models\System\Tenant;

class TenantService
{
    protected ?Tenant $tenant = null;

    public function systemConnectionName(): string
    {
        return 'system';
    }

    public function tenantConnectionName(): string
    {
        return 'tenant';
    }

    public function tenant(): ?Tenant
    {
        return $this->tenant ?? null;
    }

    public function switchTo(Tenant $tenant): void
    {
        if ($tenant->is($this->tenant)) {
            return;
        }

        settings()->flush();

        config()->set('database.connections.tenant.database', $tenant->database_uuid);

        DB::purge($this->tenantConnectionName());
        DB::reconnect($this->tenantConnectionName());

        $this->tenant = $tenant;

        URL::forceRootUrl($tenant->getCurrentAppUrl());
        URL::forceScheme('https');

        $this->updateConfigs();

        event(new TenantSwitched($this->tenant));
    }

    protected function updateConfigs(): void
    {
        config()->set([
            'app.name' => settings()->getJsonValue('texts.center', 'name'),
            'app.shortage' => $this->tenant->company_details->shortage,
        ]);

        config()->set([
            'saml.sp.assertionConsumerService.url' => route('auth.acs'),
            'saml.sp.singleLogoutService.url' => route('auth.logout'),
        ]);

        /** @var MailSmtpDTO $mailSmtp */
        $mailSmtp = settings()->getAsDTO('mail.smtp', MailSmtpDTO::class);

        $mailerName = sprintf('tenant_%s', $this->tenant->database_uuid);

        config()->set([
            'mail.default' => $mailerName,
            sprintf('mail.mailers.%s.transport', $mailerName) => 'smtp',
            sprintf('mail.mailers.%s.host', $mailerName) => $mailSmtp->host,
            sprintf('mail.mailers.%s.port', $mailerName) => $mailSmtp->port,
            sprintf('mail.mailers.%s.username', $mailerName) => $mailSmtp->username,
            sprintf('mail.mailers.%s.password', $mailerName) => $mailSmtp->password,
            'mail.from.address' => $mailSmtp->fromAddress,
            'mail.from.name' => $mailSmtp->fromName,
        ]);
    }
}
