<?php

namespace Redtree\Tenancy;

use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Queue\Events\JobProcessing;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Stringable;
use Redtree\Tenancy\Http\Controllers\SamlAuthController;
use Redtree\Tenancy\Jobs\TenantAware;
use Redtree\Tenancy\Models\System\Tenant;
use Redtree\Tenancy\Models\Tenant\Brand;
use Redtree\Tenancy\Providers\EventServiceProvider;
use Redtree\Tenancy\Services\LoggingService;
use Redtree\Tenancy\Services\MenuService;
use Redtree\Tenancy\Services\ModelClassService;
use Redtree\Tenancy\Services\PermissionsCache;
use Redtree\Tenancy\Services\SettingsService;
use Redtree\Tenancy\Services\TenantService;

class TenancyServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $this->mergeConfigFrom(__DIR__ . '/../config/tenancy.php', 'tenancy');
        $this->mergeConfigFrom(__DIR__ . '/../config/saml.php', 'saml');

        $this->app->register(EventServiceProvider::class);

        $this->app->singleton(TenantService::class, function () {
            return new TenantService();
        });
        $this->app->singleton(SettingsService::class, function () {
            return new SettingsService();
        });
        $this->app->singleton(PermissionsCache::class, function () {
            return new PermissionsCache();
        });
        $this->app->singleton(LoggingService::class, function () {
            return new LoggingService();
        });
        $this->app->singleton(ModelClassService::class, function () {
            return new ModelClassService();
        });
        $this->app->singleton(MenuService::class, function () {
            return new MenuService();
        });
    }

    public function boot(): void
    {
        $this->registerPublishes();
        $this->registerMacros();
        $this->configureQueue();

        if (config('tenancy.saml.enabled', false)) {
            Route::name('auth.')
                ->middleware('web')
                ->group(function () {
                    Route::get('login', [SamlAuthController::class, 'samlLogin'])->name('login');
                    Route::post('acs', [SamlAuthController::class, 'samlResponse'])->name('acs');
                    Route::post('logout', [SamlAuthController::class, 'samlLogout'])->name('logout')->middleware($this->samlMiddleware());
                    Route::get('slo', [SamlAuthController::class, 'slo'])->name('slo');
                });
        }

        Relation::morphMap([
            'brand' => model_class(Brand::class),
        ]);
    }

    protected function registerPublishes(): void
    {
        $this->publishes([
            __DIR__ . '/../config/tenancy.php' => config_path('tenancy.php'),
            __DIR__ . '/../config/saml.php' => config_path('saml.php'),
        ], 'config');
    }

    protected function registerMacros(): void
    {
        Stringable::macro('alpha', function () {
            return new static(preg_replace('/[^a-zA-Z]/', '', $this->value));
        });
    }

    protected function configureQueue()
    {
        app('queue')->createPayloadUsing(function ($connectionName, $queue, $payload) {
            $job = $payload['data']['command'];

            if (! $this->isTenantAware($job)) {
                return [];
            }

            return app(TenantService::class)->tenant() ? ['tenantId' => app(TenantService::class)->tenant()->id] : [];
        });

        app('events')->listen(JobProcessing::class, function (JobProcessing $event) {
            $tenantId = $event->job->payload()['tenantId'] ?? null;

            if ($tenantId === null) {
                return;
            }

            $tenant = Tenant::find($tenantId);

            if ($tenant !== null) {
                $tenant->switchTo();
            }
        });
    }

    protected function isTenantAware(object $job): bool
    {
        if ($job instanceof TenantAware) {
            return true;
        }

        return false;
    }

    protected function samlMiddleware(): string
    {
        $middleware = 'auth';

        if (config('tenancy.saml.guard') !== null) {
            $middleware .= config('tenancy.saml.guard');
        }

        return $middleware;
    }
}
