<?php

use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Support\Str;
use Redtree\Tenancy\Models\System\AssetVersion;
use Redtree\Tenancy\Models\System\Tenant;
use Redtree\Tenancy\Models\Tenant\Brand;
use Redtree\Tenancy\Services\Actions\Action;
use Redtree\Tenancy\Services\Actions\EditAction;
use Redtree\Tenancy\Services\ActionsService;
use Redtree\Tenancy\Services\LoggingService;
use Redtree\Tenancy\Services\MenuService;
use Redtree\Tenancy\Services\ModelClassService;
use Redtree\Tenancy\Services\SettingsService;
use Redtree\Tenancy\Services\TenantService;

if (! function_exists('current_tenant')) {
    function current_tenant(): ?Tenant
    {
        return app(TenantService::class)->tenant();
    }
}

if (! function_exists('settings')) {
    function settings(): SettingsService
    {
        return app(SettingsService::class);
    }
}

if (! function_exists('iolog')) {
    function iolog(): LoggingService
    {
        return app(LoggingService::class);
    }
}

if (! function_exists('base_url')) {
    function base_url(string $url, ?string $scheme = null): string
    {
        if ($scheme === null) {
            $scheme = 'https';
        }

        if (! Str::startsWith($url, ['https', 'http'])) {
            $url = $scheme . '://' . $url;
        }

        return sprintf('%s://%s', $scheme, parse_url($url, PHP_URL_HOST));
    }
}

if (! function_exists('current_user')) {
    function current_user(): ?Authenticatable
    {
        return auth()->user();
    }
}

if (! function_exists('io_encrypt')) {
    function io_encrypt(string $data, ?string $passphrase = null): string
    {
        return openssl_encrypt($data, 'aes-256-ctr', $passphrase ?? current_tenant()->uuid, 0, config('tenancy.crypt.iv'));
    }
}

if (! function_exists('io_decrypt')) {
    function io_decrypt(string $data, ?string $passphrase = null): string
    {
        return openssl_decrypt($data, 'aes-256-ctr', $passphrase ?? current_tenant()->uuid, 0, config('tenancy.crypt.iv'));
    }
}

if (! function_exists('saml_certificate')) {
    function saml_certificate(?string $path): string
    {
        if (Str::startsWith($path, '-----')) {
            return $path;
        }

        if (file_exists($path)) {
            return file_get_contents($path);
        }

        return '';
    }
}

if (! function_exists('actions')) {
    function actions(): ActionsService
    {
        return new ActionsService();
    }
}

if (! function_exists('baseAction')) {
    function baseAction(): Action
    {
        return new Action();
    }
}


if (! function_exists('editAction')) {
    function editAction(): EditAction
    {
        return new EditAction();
    }
}

if (! function_exists('current_brand')) {
    function current_brand(): Brand
    {
        return cache()
            ->store('array')
            ->rememberForever('current_brand', function () {
                return model_class(Brand::class)::where('is_default', true)->first();
            });
    }
}

if (! function_exists('asset_version')) {
    function asset_version(string $asset): string
    {
        static $versions = null;

        if ($versions === null) {
            $versions = AssetVersion::all();
        }

        $assetVersion = $versions->firstWhere('asset', $asset);

        if (! env('FORCE_RELOAD_ASSETS', false) && $assetVersion) {
            return $assetVersion->version;
        }

        return (string) time();
    }
}

if (! function_exists('current_tenant_stylesheet_url')) {
    function current_tenant_stylesheet_url(array $params = []): string
    {
        $queryParams = [
            'id' => asset_version('/css/app.css'),
        ];

        if (current_tenant() !== null) {
            $queryParams['id'] = $queryParams['id'] . preg_replace('/(-|\s|\:)/', '', optional(current_brand()->lastRevision)->updated_at);
            $queryParams['t'] = current_tenant()->uuid;
            $queryParams['b'] = current_brand()->uuid;
        }

        $queryParams = array_merge($queryParams, $params);

        return env('IO_BRANDMANAGER_URI') . '/stylesheet/io.corporate.css?' . http_build_query($queryParams);
    }
}

if (! function_exists('current_tenant_scripts_url')) {
    function current_tenant_scripts_url(): string
    {
        $queryParams = [
            'id' => asset_version('/js/app.js'),
            'app' => 'center',
        ];

        if (current_tenant() !== null) {
            $queryParams['id'] = $queryParams['id'] . preg_replace('/(-|\s|\:)/', '', optional(current_brand()->lastRevision)->updated_at);
            $queryParams['t'] = current_tenant()->uuid;
            $queryParams['b'] = current_brand()->uuid;
        }

        return env('IO_BRANDMANAGER_URI') . '/scripts/io.corporate.js?' . http_build_query($queryParams);
    }
}

if (! function_exists('current_tenant_favicon_url')) {
    function current_tenant_favicon_url(): string
    {
        $queryParams = [];

        if (current_tenant() !== null) {
            $queryParams['id'] = preg_replace('/(-|\s|\:)/', '', optional(current_brand()->lastRevision)->updated_at);
            $queryParams['t'] = current_tenant()->uuid;
            $queryParams['b'] = current_brand()->uuid;
        }

        return env('IO_BRANDMANAGER_URI') . '/favicon?' . http_build_query($queryParams);
    }
}

if (! function_exists('model_class')) {
    function model_class(string $class = '')
    {
        if (func_num_args() === 0) {
            return app(ModelClassService::class);
        }

        return app(ModelClassService::class)->getModelClass($class);
    }
}

if (! function_exists('menus')) {
    function menus(): MenuService
    {
        return app(MenuService::class);
    }
}
if (! function_exists('get_menu')) {
    function get_menu(string $key, string $app = null): array
    {
        return menus()->get($key, $app);
    }
}
